import sys
import datetime
import unicodedata
from time import mktime
from mail.catdog.catdog.src.config import cfg
from mail.catdog.catdog.src import stat

file = sys.stdout


class Holder:
    def __init__(self, postfix):
        self.postfix = postfix

    def __enter__(self):
        global file
        file = open(file=cfg().log_file + '-' + self.postfix, mode='a', buffering=8192, encoding='utf-8')
        return self

    def __exit__(self, type, value, traceback):
        global file
        file.close()

    def reopen(self):
        try:
            self.__enter__()
            return True
        except:
            return False


levels = {
    'debug': 0,
    'info': 1,
    'warning': 2,
    'error': 3,
}


def log(level, reason, message, **kwargs):
    stat.instance.log_stats.parse_line(reason=reason, message=message, **kwargs)
    if level not in levels:
        return
    if levels[level] < levels[cfg().log_level]:
        return
    line = 'tskv\ttskv_format=mail-catdog-log'
    now = datetime.datetime.now()
    line += '\ttimestamp={0}\tunixtime={1}'.format(
        now.strftime('%Y-%m-%d %H:%M:%S.%f'),
        int(now.timestamp())
    )
    line += '\tlevel={0}\treason={1}\tmessage={2}'.format(level, reason, message)
    for key, value in kwargs.items():
        line += '\t{0}={1}'.format(key, escape_value(value))
    file.write(line + '\n')


CONTROL_CATEGORIES = {'Cc', 'Cf', 'Cs', 'Co', 'Cn'}


TSKV_ESCAPE_CHARS = {
    u'\t': u'\\t',
    u'\n': u'\\n',
    u'\r': u'\\r',
    u'\\': u'\\\\',
    chr(0): u'\\0',
}


def _escape_str_value(text):
    escaped_chars = []
    for char in text:
        if char in TSKV_ESCAPE_CHARS:
            escaped_chars.append(TSKV_ESCAPE_CHARS[char])
        elif unicodedata.category(char) in CONTROL_CATEGORIES:
            escaped_chars.append(str(char.encode('unicode_escape')))
        else:
            escaped_chars.append(char)
    return ''.join(escaped_chars)


def _escape_datetime(value):
    return str(int(mktime(value.timetuple())))


ESCAPE_PLAN = {
    str: _escape_str_value,
    int: str,
    float: str,
    datetime: _escape_datetime,
    None.__class__: lambda value: ''
}


def _default_escaper(value):
    return _escape_str_value(str(value))


def escape_value(value):
    esaper = ESCAPE_PLAN.get(value.__class__, _default_escaper)
    return esaper(value)
