from enum import Enum


class BunkerException(Exception):
    def __init__(self, message, response):
        super().__init__(message)
        self.response = response


class NodeVersion(Enum):
    latest = 'latest'
    stable = 'stable'


class BunkerEndpoint(Enum):
    stable = 'http://bunker-api.yandex.net/v1'
    test = 'http://bunker-api.test.yandex.net/v1'


class Bunker(object):
    def __init__(self, session, endpoint=BunkerEndpoint.stable):
        self._session = session
        self._endpoint = endpoint.value

    async def cat(self, node, version=NodeVersion.stable):
        async with self._session.get(
            '{}/cat'.format(self._endpoint),
            params={'node': node, 'version': version.value}
        ) as r:
            assert r.status == 200, BunkerException('Failed to cat node {} ({}): {}'.format(
                node, r.status, await r.text()), r)

            if is_json(r):
                return await r.json()

            return await r.text()


def is_json(response):
    content_type_raw = response.headers.get('content-type', '')

    if content_type_raw == '':
        return False

    if content_type_raw.split('; ')[0] == 'application/json':
        return True

    return False
