package ru.yandex.mail.cerberus.asyncdb;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.jdbi.v3.core.Jdbi;
import org.jdbi.v3.core.argument.Arguments;
import org.jdbi.v3.core.array.SqlArrayTypes;
import org.jdbi.v3.core.collector.JdbiCollectors;
import org.jdbi.v3.core.mapper.ColumnMappers;
import org.jdbi.v3.core.statement.SqlLogger;
import org.jdbi.v3.core.statement.SqlStatements;
import org.jdbi.v3.postgres.PostgresPlugin;
import org.jdbi.v3.sqlobject.HandlerDecorators;
import org.jdbi.v3.sqlobject.SqlObjectPlugin;
import ru.yandex.mail.cerberus.asyncdb.internal.AsStringArgumentFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.AsStringColumnMapperFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.JsonArgumentFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.JsonBArgumentFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.JsonBColumnMapperFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.JsonColumnMapperFactory;
import ru.yandex.mail.cerberus.asyncdb.internal.decorators.ExceptionDecorator;
import ru.yandex.mail.cerberus.asyncdb.internal.value.*;
import ru.yandex.mail.cerberus.asyncdb.util.OneToManyCollectorFactory;

import javax.sql.DataSource;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

public class RepositoryFactory {
    private final Jdbi jdbi;
    private final Map<Class<? extends Repository>, Repository> repositories;

    private static int clampSeconds(Duration timeout) {
        return Math.max((int) timeout.toSeconds(), 1);
    }

    public RepositoryFactory(DataSource dataSource, ObjectMapper objectMapper, Optional<Duration> defaultQueryTimeout,
                             Optional<SqlLogger> sqlLogger) {
        this.jdbi = Jdbi.create(dataSource);
        this.repositories = new ConcurrentHashMap<>();

        jdbi.installPlugin(new SqlObjectPlugin())
            .installPlugin(new PostgresPlugin());

        jdbi.getConfig(Arguments.class)
            .register(new JsonArgumentFactory(objectMapper))
            .register(new JsonBArgumentFactory(objectMapper))
            .register(new AsStringArgumentFactory())
            .register(new ValueTypeArgumentFactory())
            .register(new LongValueTypeArgumentFactory());

        jdbi.getConfig(ColumnMappers.class)
            .register(new JsonColumnMapperFactory(objectMapper))
            .register(new JsonBColumnMapperFactory(objectMapper))
            .register(new AsStringColumnMapperFactory())
            .register(new ValueTypeColumnMapperFactory())
            .register(new LongValueTypeColumnMapperFactory());

        jdbi.getConfig(SqlArrayTypes.class)
            .register(new LongValueTypeSqlArrayTypeFactory())
            .register(new ValueTypeSqlArrayTypeFactory());

        jdbi.getConfig(HandlerDecorators.class)
            .register(new ExceptionDecorator(dataSource));

        jdbi.getConfig(JdbiCollectors.class)
            .register(new OneToManyCollectorFactory());

        defaultQueryTimeout.ifPresent(timeout -> {
            jdbi.getConfig(SqlStatements.class)
                .setQueryTimeout(clampSeconds(timeout));
        });

        sqlLogger.ifPresent(jdbi::setSqlLogger);
    }

    @SuppressWarnings("unchecked")
    public <T extends Repository> T createRepository(Class<T> repositoryClass) {
        return (T) repositories.computeIfAbsent(repositoryClass, type -> jdbi.onDemand(repositoryClass));
    }
}
