package ru.yandex.mail.cerberus.asyncdb.internal;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.val;
import org.jdbi.v3.core.argument.Argument;
import org.jdbi.v3.core.argument.ArgumentFactory;
import org.jdbi.v3.core.config.ConfigRegistry;
import org.postgresql.util.PGobject;

import java.lang.reflect.Type;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Optional;

@AllArgsConstructor
class BaseJsonArgumentFactory implements ArgumentFactory {
    private static final Argument NULL_ARGUMENT = (position, statement, ctx) -> {
        statement.setNull(position, Types.OTHER);
    };

    private final String jsonType;
    private final ObjectMapper objectMapper;

    private static void setObject(PreparedStatement statement, int position, String value, String jsonType) throws SQLException {
        val object = new PGobject();
        object.setType(jsonType);
        object.setValue(value);
        statement.setObject(position, object);
    }

    private Argument createArgument(Object value) {
        if (value == null) {
            return NULL_ARGUMENT;
        }

        if (value instanceof Optional) {
            val optionalValue = (Optional) value;
            if (optionalValue.isEmpty()) {
                return NULL_ARGUMENT;
            }

            value = optionalValue.get();
        }

        if (value instanceof String) {
            val stringValue = (String) value;
            return (position, statement, ctx) -> setObject(statement, position, stringValue, jsonType);
        } else {
            val objectValue = value;
            return (position, statement, ctx) -> {
                try {
                    setObject(statement, position, objectMapper.writeValueAsString(objectValue), jsonType);
                } catch (JsonProcessingException e) {
                    throw new SQLException(e);
                }
            };
        }
    }

    @Override
    public Optional<Argument> build(Type type, Object value, ConfigRegistry config) {
        return Optional.of(createArgument(value));
    }
}
