package ru.yandex.mail.cerberus.asyncdb.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.val;
import org.jdbi.v3.core.config.ConfigRegistry;
import org.jdbi.v3.core.mapper.ColumnMapper;
import org.jdbi.v3.core.mapper.ColumnMapperFactory;
import org.postgresql.util.PGobject;
import ru.yandex.mail.cerberus.asyncdb.exception.UnexpectedColumnTypeException;

import java.lang.reflect.Type;
import java.sql.SQLException;
import java.util.Optional;

import static org.jdbi.v3.core.generic.GenericTypes.findGenericParameter;
import static org.jdbi.v3.core.generic.GenericTypes.getErasedType;

@AllArgsConstructor
class BaseJsonColumnMapperFactory implements ColumnMapperFactory {
    private final ObjectMapper objectMapper;
    private final String jsonType;

    @Override
    public Optional<ColumnMapper<?>> build(Type type, ConfigRegistry config) {
        val erasedType = getErasedType(type);
        val valueType = findGenericParameter(type, Optional.class).orElse(type);
        val erasedValueType = getErasedType(valueType);

        val isOptional = erasedType == Optional.class;
        val isStringValue = erasedValueType == String.class;

        val javaType = objectMapper.constructType(valueType);
        val objectReader = objectMapper.readerFor(javaType);

        return Optional.of((resultSet, columnNumber, ctx) -> {
            try {
                val object = (PGobject) resultSet.getObject(columnNumber);

                if (object == null || object.getValue() == null) {
                    if (isOptional) {
                        return Optional.empty();
                    } else {
                        return null;
                    }
                }

                val value = object.getValue();
                val objectType = object.getType();
                if (!objectType.equals(jsonType)) {
                    throw new UnexpectedColumnTypeException(jsonType, objectType, resultSet.getNString(columnNumber));
                }

                val result = isStringValue ? value : objectReader.readValue(value);
                return isOptional ? Optional.of(result) : result;
            } catch (Exception e) {
                throw new SQLException(e);
            }
        });
    }
}
