package ru.yandex.mail.cerberus.asyncdb.internal;

import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import org.jdbi.v3.core.argument.internal.PojoPropertyArguments;
import org.jdbi.v3.core.config.ConfigRegistry;
import org.jdbi.v3.core.internal.IterableLike;
import org.jdbi.v3.core.mapper.reflect.internal.BeanPropertiesFactory;
import org.jdbi.v3.core.mapper.reflect.internal.PojoProperties;
import org.jdbi.v3.core.statement.SqlStatement;
import org.jdbi.v3.core.statement.SqlStatements;
import org.jdbi.v3.core.statement.StatementContext;
import org.jdbi.v3.core.statement.UnableToCreateStatementException;
import org.jdbi.v3.sqlobject.customizer.SqlStatementCustomizerFactory;
import org.jdbi.v3.sqlobject.customizer.SqlStatementParameterCustomizer;
import org.jdbi.v3.sqlobject.internal.ParameterUtil;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.List;

class CustomBeanPropertyArguments extends PojoPropertyArguments {
    CustomBeanPropertyArguments(Object obj, PojoProperties<?> properties, ConfigRegistry config) {
        super(null, obj, properties, config);
    }
}

public class BindBeanValuesFactory implements SqlStatementCustomizerFactory {
    private static UnableToCreateStatementException argumentError(String propertyName, Type beanType, StatementContext context) {
        return new UnableToCreateStatementException("Unable to get " + propertyName + " argument for " + beanType, context);
    }

    private static PojoProperties<?> findProperties(List<?> beans, ConfigRegistry config) {
        val bean = beans.get(0);
        return BeanPropertiesFactory.propertiesFor(bean.getClass(), config);
    }

    // bind as (:name_a1, :name_b1), (:name_a2, :name_b2), ..., (:name_aN, :name_bN)
    private static void bindBeanValues(String argName, List<?> beans, SqlStatement<?> statement) {
        val context = statement.getContext();
        val config = context.getConfig();
        val properties = findProperties(beans, config);

        val parser = statement.getConfig().get(SqlStatements.class).getSqlParser();

        val propertyArgNameByPropertyName = EntryStream.of(properties.getProperties())
            .mapValues(property -> argName + '_' + property.getName())
            .toMap();

        val namesListString = IntStreamEx.range(0, beans.size())
            .mapToEntry(Integer::valueOf, beans::get)
            .mapValues(bean -> new CustomBeanPropertyArguments(bean, properties, config))
            .mapKeyValue((index, beanPropertyArguments) -> {
                return EntryStream.of(propertyArgNameByPropertyName)
                    .mapValues(name -> name + index)
                    .peekKeyValue((propertyName, propertyArgName) -> {
                        val argument = beanPropertyArguments.find(propertyName, context)
                            .orElseThrow(() -> argumentError(propertyName, properties.getType(), context));
                        statement.bind(propertyArgName, argument);
                    })
                    .values()
                    .map(name -> parser.nameParameter(name, context))
                    .joining(", ", "(", ")");
            })
            .joining(", ");

        val columns = EntryStream.of(properties.getProperties())
            .keys()
            .joining(", ");

        statement.define(argName, namesListString);
        statement.define(argName + "_columns", columns);
    }

    @Override
    public SqlStatementParameterCustomizer createForParameter(Annotation annotation, Class<?> sqlObjectType, Method method,
                                                              Parameter param, int index, Type paramType) {
        final String name = ParameterUtil.findParameterName("", param)
            .orElseThrow(() -> new UnsupportedOperationException("A @BindBeanValues parameter was not given a name, "
                + "and parameter name data is not present in the class file, for: "
                + param.getDeclaringExecutable() + "::" + param));

        return (stmt, arg) -> {
            if (arg == null) {
                throw new IllegalArgumentException("@BindBeanValues doesn't support null collections");
            } else {
                val list = IterableLike.toList(arg);
                if (list.isEmpty()) {
                    throw new IllegalArgumentException("@BindBeanValues doesn't support empty collection");
                }
                bindBeanValues(name, list, stmt);
            }
        };
    }
}
