package ru.yandex.mail.cerberus.asyncdb.internal;

import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.StreamEx;
import org.jdbi.v3.core.generic.GenericTypes;
import org.jdbi.v3.core.internal.IterableLike;
import org.jdbi.v3.sqlobject.customizer.SqlStatementCustomizerFactory;
import org.jdbi.v3.sqlobject.customizer.SqlStatementParameterCustomizer;
import ru.yandex.mail.cerberus.asyncdb.SortOrder;
import ru.yandex.mail.cerberus.asyncdb.annotations.BindBeans;

import java.beans.FeatureDescriptor;
import java.beans.Introspector;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.Comparator;
import java.util.List;

public class BindBeansFactory implements SqlStatementCustomizerFactory {
    private static UnsupportedOperationException newInvalidTypeError() {
        return new UnsupportedOperationException("A @BindBeans requires Iterable<?> argument to be applied");
    }

    public static Comparator<String> resolveComparator(SortOrder order) {
        return order == SortOrder.ASC ? Comparator.naturalOrder() : Comparator.reverseOrder();
    }

    @Override
    @SneakyThrows
    public SqlStatementParameterCustomizer createForParameter(Annotation annotation,
                                                              Class<?> sqlObjectType,
                                                              Method method,
                                                              Parameter param,
                                                              int index,
                                                              Type type) {
        val sortOrder = ((BindBeans) annotation).propertiesOrder();

        if (!param.isNamePresent()) {
            throw new UnsupportedOperationException("A @BindBeans parameter name not found. Enable -parameters javac option and retry");
        }

        if (!Iterable.class.isAssignableFrom(param.getType())) {
            throw newInvalidTypeError();
        }

        val elementType = IterableLike.elementTypeOf(type)
            .map(GenericTypes::getErasedType)
            .orElseThrow(BindBeansFactory::newInvalidTypeError);

        val beanInfo = Introspector.getBeanInfo(elementType, Object.class);
        val properties = StreamEx.of(beanInfo.getPropertyDescriptors())
            .map(FeatureDescriptor::getName)
            .sorted(resolveComparator(sortOrder))
            .toImmutableList();

        val name = param.getName();
        return (stmt, arg) -> {
            if (arg == null) {
                throw new IllegalArgumentException("argument is null; null was explicitly forbidden on BindBeans");
            }

            val values = (arg instanceof List) ? (List) arg : IterableLike.toList(arg);
            stmt.bindBeanList(name, values, properties);
        };
    }
}
