package ru.yandex.mail.cerberus.asyncdb.internal;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jdbi.v3.core.config.ConfigRegistry;
import org.jdbi.v3.core.mapper.RowMappers;
import org.jdbi.v3.core.mapper.reflect.ConstructorMapper;
import org.jdbi.v3.sqlobject.config.Configurer;
import ru.yandex.mail.cerberus.asyncdb.annotations.CompositeId;

import java.lang.annotation.Annotation;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

import static org.jdbi.v3.core.generic.GenericTypes.getErasedType;
import static ru.yandex.mail.cerberus.asyncdb.internal.RepositoryInfoUtils.findEntityType;
import static ru.yandex.mail.cerberus.asyncdb.internal.RepositoryInfoUtils.findIdType;

@Slf4j
public class CrudRepositoryConfigurer implements Configurer {
    private static final Map<Class<?>, Consumer<ConfigRegistry>> CONFIGURER_CACHE = new ConcurrentHashMap<>();

    private static Consumer<ConfigRegistry> makeConfigurer(Class<?> sqlObjectType) {
        log.info("Register repo: {}", sqlObjectType);

        val idType = getErasedType(findIdType(sqlObjectType));
        val entityType = getErasedType(findEntityType(sqlObjectType));
        val entityRowMapper = ConstructorMapper.factory(entityType);

        final var idClassOpt = sqlObjectType.isAnnotationPresent(CompositeId.class)
            ? Optional.of(idType)
            : Optional.<Class<?>>empty();
        val idRowMapperOpt = idClassOpt.map(ConstructorMapper::factory);

        return registry -> {
            val rowMappers = registry.get(RowMappers.class);
            if (rowMappers.findFor(entityType).isEmpty()) {
                rowMappers.register(entityRowMapper);
            }

            idClassOpt.ifPresent(idClass -> {
                if (rowMappers.findFor(idClass).isEmpty()) {
                    idRowMapperOpt.ifPresent(rowMappers::register);
                }
            });
        };
    }

    @Override
    public void configureForType(ConfigRegistry registry, Annotation annotation, Class<?> sqlObjectType) {
        val configurer = CONFIGURER_CACHE.computeIfAbsent(sqlObjectType, CrudRepositoryConfigurer::makeConfigurer);
        configurer.accept(registry);
    }
}
