package ru.yandex.mail.cerberus.asyncdb.internal.decorators;

import lombok.Lombok;
import lombok.val;
import org.jdbi.v3.core.JdbiException;
import org.jdbi.v3.core.collector.ElementTypeNotFoundException;
import org.jdbi.v3.core.collector.NoSuchCollectorException;
import org.jdbi.v3.core.extension.NoSuchExtensionException;
import org.jdbi.v3.core.mapper.MappingException;
import org.jdbi.v3.core.mapper.NoSuchMapperException;
import org.jdbi.v3.core.result.NoResultsException;
import org.jdbi.v3.core.result.ResultSetException;
import org.jdbi.v3.core.result.UnableToProduceResultException;
import org.jdbi.v3.core.statement.StatementException;
import org.jdbi.v3.core.statement.UnableToCreateStatementException;
import org.jdbi.v3.core.statement.UnableToExecuteStatementException;
import org.jdbi.v3.core.transaction.TransactionException;
import org.jdbi.v3.core.transaction.UnableToManipulateTransactionIsolationLevelException;
import org.jdbi.v3.core.transaction.UnableToRestoreAutoCommitStateException;
import org.jdbi.v3.sqlobject.Handler;
import org.jdbi.v3.sqlobject.HandlerDecorator;
import org.jdbi.v3.sqlobject.UnableToCreateSqlObjectException;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.InvalidDataAccessApiUsageException;
import org.springframework.dao.InvalidDataAccessResourceUsageException;
import org.springframework.dao.TypeMismatchDataAccessException;
import org.springframework.dao.UncategorizedDataAccessException;
import org.springframework.jdbc.support.SQLErrorCodeSQLExceptionTranslator;
import org.springframework.jdbc.support.SQLExceptionTranslator;
import org.springframework.transaction.TransactionSystemException;

import javax.sql.DataSource;
import java.lang.reflect.Method;
import java.sql.SQLException;

public class ExceptionDecorator implements HandlerDecorator {
    private final SQLExceptionTranslator translator;

    public ExceptionDecorator(DataSource dataSource) {
        this.translator = new SQLErrorCodeSQLExceptionTranslator(dataSource);
    }

    private static DataAccessException translateJdbiNonSQLException(JdbiException e) {
        if (e instanceof ElementTypeNotFoundException) {
            return new TypeMismatchDataAccessException("Element type not found", e);
        } else if (e instanceof MappingException) {
            return new TypeMismatchDataAccessException("Mapping failed", e);
        } else if (e instanceof NoResultsException) {
            return new EmptyResultDataAccessException("Empty result set", 0, e);
        } else if (e instanceof NoSuchCollectorException) {
            return new InvalidDataAccessApiUsageException("No such collector", e);
        } else if (e instanceof NoSuchExtensionException) {
            return new InvalidDataAccessApiUsageException("No such extension", e);
        } else if (e instanceof NoSuchMapperException) {
            return new InvalidDataAccessApiUsageException("No such mapper", e);
        } else if (e instanceof ResultSetException) {
            return new InvalidDataAccessResourceUsageException("Result set error", e);
        } else if (e instanceof UnableToCreateSqlObjectException) {
            return new InvalidDataAccessApiUsageException("Sql object error", e);
        } else if (e instanceof UnableToCreateStatementException) {
            return new InvalidDataAccessApiUsageException("Unable to create statement", e);
        } else if (e instanceof UnableToExecuteStatementException) {
            return new InvalidDataAccessApiUsageException("Unable to execute statement", e);
        } else if (e instanceof UnableToProduceResultException) {
            return new InvalidDataAccessApiUsageException("Unable to produce result", e);
        } else if (e instanceof StatementException) {
            return new InvalidDataAccessApiUsageException("Statement error", e);
        } else {
            return new UncategorizedDataAccessException("Unknown error", e) {};
        }
    }

    private static org.springframework.transaction.TransactionException translateJdbiTransactionException(JdbiException e) {
        if (e instanceof TransactionException) {
            return new TransactionSystemException("Transaction error", e);
        } else if (e instanceof UnableToManipulateTransactionIsolationLevelException) {
            return new TransactionSystemException("Unable to manipulate transaction isolation level", e);
        } else if (e instanceof UnableToRestoreAutoCommitStateException) {
            return new TransactionSystemException("Unable to restore auto commit state", e);
        } else {
            return null;
        }
    }

    private DataAccessException translateJdbiException(String name, JdbiException e) {
        if (e.getCause() instanceof SQLException) {
            return translator.translate(name, null, (SQLException) e.getCause());
        } else {
            return translateJdbiNonSQLException(e);
        }
    }

    @Override
    public Handler decorateHandler(Handler base, Class<?> sqlObjectType, Method method) {
        val name = method.getName();

        return (target, args, handle) -> {
            try {
                return base.invoke(target, args, handle);
            } catch (JdbiException e) {
                val txException = translateJdbiTransactionException(e);
                if (txException != null) {
                    throw txException;
                }

                throw translateJdbiException(name, e);
            } catch (Throwable e) {
                throw Lombok.sneakyThrow(e);
            }
        };
    }
}
