package ru.yandex.mail.cerberus.asyncdb.util;

import lombok.val;
import org.jdbi.v3.core.collector.CollectorFactory;
import org.jdbi.v3.core.generic.GenericTypes;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collector;
import java.util.stream.Collector.Characteristics;

class OneToManyBuilder<K, V> {
    private final Map<K, List<V>> map = new HashMap<>();

    private List<V> values(K key) {
        return map.computeIfAbsent(key, k -> new ArrayList<>());
    }

    void put(Map.Entry<K, V> entry) {
        val values = values(entry.getKey());
        val value = entry.getValue();

        if (value != null) {
            values.add(value);
        }
    }

    OneToManyBuilder<K, V> combine(OneToManyBuilder<K, V> builder) {
        builder.map.forEach((k, v) -> values(k).addAll(v));
        return this;
    }

    OneToMany<K, V> build() {
        return new OneToMany<>(map);
    }
}

public class OneToManyCollectorFactory implements CollectorFactory {
    private final TypeVariable<Class<OneToMany>> mapKey;
    private final TypeVariable<Class<OneToMany>> mapValue;

    private static <K, V> Collector<Map.Entry<K, V>, OneToManyBuilder<K, V>, OneToMany<K, V>> oneToManyCollector() {
        return Collector.of(
            OneToManyBuilder::new,
            OneToManyBuilder::put,
            OneToManyBuilder::combine,
            OneToManyBuilder::build,
            Characteristics.UNORDERED);
    }

    public OneToManyCollectorFactory() {
        val mapParams = OneToMany.class.getTypeParameters();
        mapKey = mapParams[0];
        mapValue = mapParams[1];
    }

    @Override
    public boolean accepts(Type containerType) {
        return containerType instanceof ParameterizedType && OneToMany.class.equals(GenericTypes.getErasedType(containerType));
    }

    @Override
    public Optional<Type> elementType(Type containerType) {
        Class<?> erasedType = GenericTypes.getErasedType(containerType);

        if (OneToMany.class.isAssignableFrom(erasedType)) {
            Type keyType = GenericTypes.resolveType(mapKey, containerType);
            Type valueType = GenericTypes.resolveType(mapValue, containerType);
            return java.util.Optional.of(GenericTypes.resolveMapEntryType(keyType, valueType));
        }

        return Optional.empty();
    }

    @Override
    public Collector<?, ?, ?> build(Type containerType) {
        return oneToManyCollector();
    }
}
