package ru.yandex.mail.cerberus.client.ops;

import io.micronaut.http.annotation.Body;
import io.micronaut.http.annotation.Delete;
import io.micronaut.http.annotation.Get;
import io.micronaut.http.annotation.Header;
import io.micronaut.http.annotation.Post;
import io.micronaut.http.annotation.QueryValue;
import io.micronaut.http.annotation.Status;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.client.dto.Resource;
import ru.yandex.mail.cerberus.client.dto.ResourceData;
import ru.yandex.mail.cerberus.client.dto.ResourceType;
import ru.yandex.mail.micronaut.common.RawJsonString;

import javax.annotation.Nullable;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static io.micronaut.http.HttpStatus.CREATED;
import static io.micronaut.http.HttpStatus.NO_CONTENT;
import static ru.yandex.mail.cerberus.client.Headers.X_CONSISTENCY_TARGET;

public interface ResourceOperations {
    @Status(CREATED)
    @Post("/type/create")
    Mono<ResourceType> createResourceType(@Body @Valid ResourceType type);

    @Post("/type/get_or_create")
    Mono<ResourceType> getOrCreateResourceType(@Body @Valid ResourceType type);

    @Get("/type")
    Mono<Optional<ResourceType>> findType(@QueryValue @Valid ResourceTypeName name,
                                          @Header(X_CONSISTENCY_TARGET) @Nullable ReadTarget readTarget);

    @Get("/types")
    Mono<Map<ResourceTypeName, ResourceType>> findTypes(@QueryValue @NotEmpty Set<ResourceTypeName> names,
                                                        @Header(X_CONSISTENCY_TARGET) @Nullable ReadTarget readTarget);

    @Get("/all")
    Mono<Page<ResourceId, Resource<RawJsonString>>> findResourcesByType(@QueryValue @Valid ResourceTypeName type,
                                                                        @QueryValue("page_size") int pageSize,
                                                                        @QueryValue("page_id") @Nullable @Valid ResourceId pageId,
                                                                        @Header(X_CONSISTENCY_TARGET) @Nullable ReadTarget readTarget);

    @Status(CREATED)
    @Post("/create")
    Mono<Resource<RawJsonString>> createResource(@Body @Valid ResourceData<RawJsonString> data);

    @Status(CREATED)
    @Post("/create/bulk")
    Mono<List<Resource<RawJsonString>>> createResources(@Body @Valid List<ResourceData<RawJsonString>> data);

    @Status(CREATED)
    @Post("/add")
    Mono<Resource<RawJsonString>> addResource(@QueryValue("skip_existing") @Nullable Boolean skipExisting,
                                              @Body @Valid Resource<RawJsonString> resource);

    @Status(CREATED)
    @Post("/add/bulk")
    Mono<List<Resource<RawJsonString>>> addResources(@QueryValue("skip_existing") @Nullable Boolean skipExisting,
                                                     @Body @Valid List<Resource<RawJsonString>> resources);

    @Post("/update")
    @Status(NO_CONTENT)
    Mono<Void> updateResource(@Body @Valid Resource<RawJsonString> resource);

    @Post("/update/bulk")
    @Status(NO_CONTENT)
    Mono<Void> updateResources(@Body @Valid List<Resource<RawJsonString>> resources);

    @Delete
    @Status(NO_CONTENT)
    Mono<Void> deleteResource(@QueryValue @Nullable Boolean strict,
                              @QueryValue @Valid ResourceTypeName type,
                              @QueryValue @Valid ResourceId id);

    @Delete("/batch")
    @Status(NO_CONTENT)
    Mono<Void> deleteResources(@QueryValue @Nullable Boolean strict,
                               @QueryValue @Valid ResourceTypeName type,
                               @Body @Valid Set<ResourceId> id);
}
