package ru.yandex.mail.cerberus.core;

import lombok.AllArgsConstructor;
import lombok.Lombok;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.micronaut.common.CerberusUtils;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.asyncdb.CrudRepository;
import ru.yandex.mail.cerberus.asyncdb.RoCrudRepository;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.dao.tx.TxManager;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.findMissing;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@AllArgsConstructor
public class CrudManager<ID, Entity> {
    private final CrudRepository<ID, Entity> repository;
    private final RoCrudRepository<ID, Entity> roRepository;
    private final TxManagerGroup txManagerGroup;
    private final ChangeLog changeLog;
    private final EntityInfoProvider<ID, Entity> entityInfoProvider;

    public TxManager writingTxManager() {
        return txManagerGroup.getWriting();
    }

    public TxManager readingTxManager(ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget);
    }

    private RoCrudRepository<ID, Entity> readingRepo(ReadTarget readTarget) {
        return readTarget == ReadTarget.MASTER ? repository : roRepository;
    }

    public Entity insertSync(Entity entity) {
        return writingTxManager().execute(() -> {
            val insertedEntity = repository.insert(entity);
            changeLog.writeCreation(insertedEntity, entityInfoProvider.subjectExtractor());
            return insertedEntity;
        });
    }

    public CompletableFuture<Entity> insert(Entity entity) {
        return writingTxManager().executeAsync(() -> insertSync(entity));
    }

    public List<Entity> insertSync(CollisionStrategy collisionStrategy, List<Entity> entities) {
        if (entities.isEmpty()) {
            return emptyList();
        }

        val ids = mapToList(entities, entityInfoProvider::getId);

        return writingTxManager().execute(() -> {
            val existingIds = repository.findExistingIds(ids);

            if (!existingIds.isEmpty() && collisionStrategy == CollisionStrategy.FAIL) {
                throw entityInfoProvider.alreadyExistsException(existingIds);
            }

            val newEntities = existingIds.isEmpty()
                ? entities
                : StreamEx.of(entities)
                    .remove(entity -> existingIds.contains(entityInfoProvider.getId(entity)))
                    .toImmutableList();

            if (newEntities.isEmpty()) {
                return emptyList();
            }

            val insertedEntities = repository.insertAll(newEntities);
            changeLog.writeCreation(insertedEntities, entityInfoProvider.subjectExtractor());
            return insertedEntities;
        });
    }

    public CompletableFuture<List<Entity>> insert(CollisionStrategy collisionStrategy, List<Entity> entities) {
        return writingTxManager().executeAsync(() -> insertSync(collisionStrategy, entities));
    }

    public CompletableFuture<Void> update(Entity entity) {
        return writingTxManager().runAsync(() -> {
            val previousEntity = repository.update(entity)
                .orElseThrow(() -> {
                    val id = entityInfoProvider.getId(entity);
                    return entityInfoProvider.notFoundException(singletonList(id));
                });
            changeLog.writeUpdating(previousEntity, entity, entityInfoProvider.subjectExtractor());
        });
    }

    public CompletableFuture<Void> update(Collection<Entity> entities) {
        return writingTxManager().runAsync(() -> {
            val previousEntities = repository.updateAll(entities);
            if (previousEntities.size() != entities.size()) {
                val missingIds = CerberusUtils.findMissing(
                    mapToSet(previousEntities, entityInfoProvider::getId),
                    mapToList(entities, entityInfoProvider::getId));
                throw Lombok.sneakyThrow(entityInfoProvider.notFoundException(missingIds));
            }
            changeLog.writeUpdating(previousEntities, entities, entityInfoProvider.subjectExtractor());
        });
    }

    public Page<ID, Entity> findAllSync(Pageable<ID> pageable, ReadTarget readTarget) {
        return readingTxManager(readTarget).execute(() -> {
            return readingRepo(readTarget).findPage(pageable, entityInfoProvider::getId);
        });
    }

    public CompletableFuture<Page<ID, Entity>> findAll(Pageable<ID> pageable, ReadTarget readTarget) {
        return readingTxManager(readTarget).executeAsync(() -> findAllSync(pageable, readTarget));
    }

    public CompletableFuture<Void> deleteById(ID id, DeletionMode mode) {
        return writingTxManager().runAsync(() -> {
            if (repository.delete(id)) {
                changeLog.writeDeletion(id, entityInfoProvider.idSubjectExtractor());
            } else if (mode == DeletionMode.STRICT) {
                throw entityInfoProvider.notFoundException(singletonList(id));
            }
        });
    }

    public CompletableFuture<Void> deleteById(Collection<ID> ids, DeletionMode mode) {
        return writingTxManager().runAsync(() -> {
            val deletedIds = repository.deleteAll(ids);

            if (mode == DeletionMode.STRICT && deletedIds.size() != ids.size()) {
                val missingIds = findMissing(Set.copyOf(ids), deletedIds);
                throw entityInfoProvider.notFoundException(missingIds);
            }

            if (!deletedIds.isEmpty()) {
                changeLog.writeDeletion(deletedIds, entityInfoProvider.idSubjectExtractor());
            }
        });
    }

    public CompletableFuture<Void> delete(Entity entity, DeletionMode mode) {
        return deleteById(entityInfoProvider.getId(entity), mode);
    }

    public CompletableFuture<Void> delete(Collection<Entity> entities, DeletionMode mode) {
        return deleteById(mapToList(entities, entityInfoProvider::getId), mode);
    }
}
