package ru.yandex.mail.cerberus.core.change_log;

import java.time.OffsetDateTime;
import java.util.Collection;
import java.util.List;
import java.util.OptionalLong;

import javax.inject.Inject;
import javax.inject.Singleton;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;

import ru.yandex.mail.cerberus.dao.change_log.ChangeAction;
import ru.yandex.mail.cerberus.dao.change_log.ChangeLogEntity;
import ru.yandex.mail.diffusion.Diffusion;
import ru.yandex.mail.diffusion.json.JsonPatchCollector;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.micronaut.common.context.ContextManager;

import static ru.yandex.mail.cerberus.dao.DaoConstants.nextval;
import static ru.yandex.mail.micronaut.common.CerberusUtils.foldToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
class RequestInfo {
    String requestId;
    OptionalLong actorUid;
    String actorServiceName;
}

@Singleton
@Slf4j
public class PersistentChangeLog implements ChangeLog {
    private static final String UNKNOWN_ACTOR_SERVICE = "unknown";

    private final ObjectMapper objectMapper;
    private final JsonMapper jsonMapper;
    private final Diffusion diffusion;

    @Inject
    public PersistentChangeLog(ObjectMapper objectMapper, JsonMapper jsonMapper) {
        this.objectMapper = objectMapper;
        this.jsonMapper = jsonMapper;
        this.diffusion = new Diffusion();
    }

    private static RequestInfo collectRequestInfo() {
        val requestId = ContextManager.currentRequestId();
        return ContextManager.currentRequestAuthenticationInfo()
                .map(info -> new RequestInfo(requestId, info.userUid(), info.clientName()))
                .orElseGet(() -> new RequestInfo(requestId, OptionalLong.empty(), UNKNOWN_ACTOR_SERVICE));
    }

    private ChangeLogEntity createEntity(ChangeSubject subject, ChangeAction action, String change,
                                         RequestInfo requestInfo) {
        val subjectJson = jsonMapper.toJson(subject);
        return new ChangeLogEntity(nextval(), action, subject.changeType(), subjectJson, OffsetDateTime.now(),
                requestInfo.getActorUid(), requestInfo.getActorServiceName(), requestInfo.getRequestId(), change);
    }

    private <T> ChangeLogEntity newCreationChange(T object, SubjectExtractor<T> subjectExtractor) {
        val change = jsonMapper.toJson(object);
        return createEntity(subjectExtractor.extractSubject(object), ChangeAction.ADD, change, collectRequestInfo());
    }

    private <T> ChangeLogEntity newUpdatingChange(T oldObject, T newObject, SubjectExtractor<T> subjectExtractor) {
        if (oldObject.getClass() != newObject.getClass()) {
            throw new IllegalArgumentException("Objects type mismatch: old is '" + oldObject.getClass()
                    + "', but new is '" + newObject.getClass() + "'");
        }

        val oldSubject = subjectExtractor.extractSubject(oldObject);
        val newSubject = subjectExtractor.extractSubject(newObject);
        if (!oldSubject.equals(newSubject)) {
            throw new IllegalArgumentException("Invalid updating change, old and new objects have different subjects");
        }

        try (val collector = new JsonPatchCollector(objectMapper)) {
            @SuppressWarnings("unchecked")
            val matcher = diffusion.fieldMatcherFor((Class<T>) oldObject.getClass());
            val change = collector.collect(oldObject, newObject, matcher);
            return createEntity(oldSubject, ChangeAction.UPDATE, change, collectRequestInfo());
        }
    }

    private ChangeLogEntity newDeletionChange(ChangeSubject subject) {
        return createEntity(subject, ChangeAction.DELETE, "{}", collectRequestInfo());
    }

    private void commitChange(ChangeLogEntity change) {
        try {
            log.info("commitChange:" + objectMapper.writeValueAsString(change));
        } catch (JsonProcessingException e) {
            log.warn("Error processing changelog entity: " + change.toString(), e);
        }
    }

    private void commitChanges(List<ChangeLogEntity> changes) {
        for (ChangeLogEntity change : changes) {
            commitChange(change);
        }
    }

    @Override
    public <T> void writeCreation(T object, SubjectExtractor<T> subjectExtractor) {
        commitChange(newCreationChange(object, subjectExtractor));
    }

    @Override
    public <T> void writeCreation(Collection<T> objects, SubjectExtractor<T> subjectExtractor) {
        if (objects.isEmpty()) {
            return;
        }

        final var changes = mapToList(objects, object -> newCreationChange(object, subjectExtractor));
        commitChanges(changes);
    }

    @Override
    public <T> void writeUpdating(T oldObject, T newObject, SubjectExtractor<T> subjectExtractor) {
        commitChange(newUpdatingChange(oldObject, newObject, subjectExtractor));
    }

    @Override
    public <T> void writeUpdating(Collection<T> oldObjects, Collection<T> newObjects,
                                  SubjectExtractor<T> subjectExtractor) {
        if (oldObjects.size() != newObjects.size()) {
            throw new IllegalArgumentException("oldObjects size doesn't match newObjects size");
        }

        if (oldObjects.isEmpty()) {
            return;
        }

        final var changes = foldToList(oldObjects, newObjects, (oldObj, newObj) -> {
            return newUpdatingChange(oldObj, newObj, subjectExtractor);
        });
        commitChanges(changes);
    }

    @Override
    public void writeDeletion(ChangeSubject subject) {
        commitChange(newDeletionChange(subject));
    }

    @Override
    public void writeDeletion(Collection<ChangeSubject> subjects) {
        if (subjects.isEmpty()) {
            return;
        }

        val changes = mapToList(subjects, this::newDeletionChange);
        commitChanges(changes);
    }
}
