package ru.yandex.mail.cerberus.core.grant;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.inject.Inject;
import javax.inject.Singleton;

import com.fasterxml.jackson.annotation.JsonCreator;
import io.micronaut.core.annotation.Introspected;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceKey;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.AllowedActions;
import ru.yandex.mail.cerberus.client.dto.AllowedTypeActions;
import ru.yandex.mail.cerberus.client.dto.Grant;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.core.change_log.ChangeSubject;
import ru.yandex.mail.cerberus.core.change_log.LongIdSubject;
import ru.yandex.mail.cerberus.core.resource.ResourceManager;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.grant.GrantEntity;
import ru.yandex.mail.cerberus.dao.grant.GrantRepositoryGroup;
import ru.yandex.mail.cerberus.dao.grant.ResourceActionsInfo;
import ru.yandex.mail.cerberus.dao.grant.ResourceTypeActionsInfo;
import ru.yandex.mail.cerberus.dao.tx.TxManager;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;
import ru.yandex.mail.cerberus.dao.user.UserRepositoryGroup;
import ru.yandex.mail.cerberus.exception.UnsupportedActionsException;

import static com.ea.async.Async.await;
import static java.util.function.Predicate.not;

@Value
@Introspected
@AllArgsConstructor(onConstructor_=@JsonCreator)
class GroupGrantChangeSubject implements ChangeSubject {
    GroupId groupId;
    GroupType groupType;

    GroupGrantChangeSubject(GroupKey groupKey) {
        this(groupKey.getId(), groupKey.getType());
    }

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.GROUP_GRANT;
    }
}

@Singleton
@AllArgsConstructor(onConstructor_= @Inject)
public class DefaultGrantManager implements GrantManager {
    private final GrantRepositoryGroup grantRepositoryGroup;
    private final UserRepositoryGroup userRepositoryGroup;
    private final ResourceManager resourceManager;
    private final ChangeLog changeLog;
    private final TxManagerGroup txManagerGroup;

    private TxManager writingTxManager() {
        return txManagerGroup.getWriting();
    }

    private static ChangeSubject extractSubject(GrantEntity entity, ChangeSubjectType subjectType) {
        return entity
            .getUid()
            .<ChangeSubject>map(uid -> new LongIdSubject(uid.getValue(), ChangeSubjectType.USER_GRANT))
            .or(() -> entity.getRoleId()
                .map(roleId -> new LongIdSubject(roleId.getValue(), ChangeSubjectType.ROLE_GRANT))
            )
            .or(() -> entity.extractGroupKey()
                .map(GroupGrantChangeSubject::new)
            )
            .orElseThrow();
    }

    private CompletableFuture<Void> allow(Grant grant, ChangeSubjectType changeSubjectType, Function<Grant, GrantEntity> inserter) {
        val actions = await(resourceManager.getPossibleActions(grant.getResourceType(), ReadTarget.MASTER));

        val invalidActions = StreamEx.of(grant.getActions())
            .filter(not(actions::contains))
            .toImmutableList();

        if (!invalidActions.isEmpty()) {
            throw new UnsupportedActionsException(invalidActions, actions);
        }

        return writingTxManager().runAsync(() -> {
            val entity = inserter.apply(grant);
            changeLog.writeCreation(entity, e -> extractSubject(e, changeSubjectType));
        });
    }

    private CompletableFuture<Void> deny(Supplier<Optional<ChangeSubject>> denySupplier) {
        return writingTxManager().runAsync(() -> {
            denySupplier.get().ifPresent(changeLog::writeDeletion);
        });
    }

    @Override
    public CompletableFuture<Boolean> isSuperuser(Uid uid, ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            return userRepositoryGroup.getReading(readTarget)
                .isSuperuser(uid)
                .orElse(false);
        });
    }

    @Override
    public CompletableFuture<Void> setGrant(Uid uid, Grant grant) {
        val repo = grantRepositoryGroup.getWriting();
        if (grant.getActions().isEmpty()) {
            return deny(() -> {
                return repo
                    .delete(uid, grant.getResourceType(), grant.getResourceId())
                    .map(ignore -> new LongIdSubject(uid.getValue(), ChangeSubjectType.USER_GRANT));
            });
        } else {
            return allow(grant, ChangeSubjectType.USER_GRANT, newGrant -> {
                return repo.upsert(uid, newGrant.getResourceType(), newGrant.getResourceId(), newGrant.getActions());
            });
        }
    }

    @Override
    public CompletableFuture<Void> setGrant(GroupKey groupKey, Grant grant) {
        val repo = grantRepositoryGroup.getWriting();
        if (grant.getActions().isEmpty()) {
            return deny(() -> {
                return repo
                    .delete(groupKey, grant.getResourceType(), grant.getResourceId())
                    .map(ignore -> new GroupGrantChangeSubject(groupKey));
            });
        } else {
            return allow(grant, ChangeSubjectType.GROUP_GRANT, newGrant -> {
                return repo.upsert(groupKey, newGrant.getResourceType(), newGrant.getResourceId(), newGrant.getActions());
            });
        }
    }

    @Override
    public CompletableFuture<Void> setGrant(RoleId roleId, Grant grant) {
        val repo = grantRepositoryGroup.getWriting();
        if (grant.getActions().isEmpty()) {
            return deny(() -> {
                return repo
                    .delete(roleId, grant.getResourceType(), grant.getResourceId())
                    .map(ignore -> new LongIdSubject(roleId.getValue(), ChangeSubjectType.USER_GRANT));
            });
        } else {
            return allow(grant, ChangeSubjectType.ROLE_GRANT, newGrant -> {
                return repo.upsert(roleId, newGrant.getResourceType(), newGrant.getResourceId(), newGrant.getActions());
            });
        }
    }

    private static AllowedActions.ResourceInfo convert(ResourceActionsInfo info) {
        return new AllowedActions.ResourceInfo(info.getResourceId(), info.getResourceType(), info.getActions());
    }

    @Override
    public CompletableFuture<AllowedActions> actions(Uid uid, Set<ResourceKey> keys, ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            val actionsInfo = grantRepositoryGroup.getReading(readTarget)
                .findResourceActions(uid, keys);

            val result = StreamEx.of(actionsInfo)
                .map(DefaultGrantManager::convert)
                .toImmutableList();

            return new AllowedActions(result);
        });
    }

    @Override
    public CompletableFuture<AllowedActions> actions(Uid uid, ResourceTypeName resourceType, Set<ResourceId> resources,
                                                     ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            val actionsInfo = grantRepositoryGroup.getReading(readTarget)
                .findResourceActions(uid, resourceType, resources);

            val result = StreamEx.of(actionsInfo)
                .map(DefaultGrantManager::convert)
                .toImmutableList();

            return new AllowedActions(result);
        });
    }

    private static AllowedTypeActions.ResourceTypeInfo convert(ResourceTypeActionsInfo info) {
        return new AllowedTypeActions.ResourceTypeInfo(info.getResourceType(), info.getActions());
    }

    @Override
    public CompletableFuture<AllowedTypeActions> typesActions(Uid uid, Set<ResourceTypeName> resourceTypes,
                                                              ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            val actionsInfo = grantRepositoryGroup.getReading(readTarget)
                .findResourceTypeActions(uid, resourceTypes);

            final var result = StreamEx.of(actionsInfo)
                .map(DefaultGrantManager::convert)
                .toImmutableList();

            return new AllowedTypeActions(result);
        });
    }
}
