package ru.yandex.mail.cerberus.core.group;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Singleton;

import com.fasterxml.jackson.annotation.JsonCreator;
import io.micronaut.core.annotation.Introspected;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.Group;
import ru.yandex.mail.cerberus.client.dto.GroupData;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.CrudManager;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.core.EntityInfoProvider;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.core.change_log.ChangeSubject;
import ru.yandex.mail.cerberus.core.change_log.SubjectExtractor;
import ru.yandex.mail.cerberus.core.mapper.GroupMapper;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.group.GroupEntity;
import ru.yandex.mail.cerberus.dao.group.GroupRepositoryGroup;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;
import ru.yandex.mail.cerberus.dao.user.UserRepository;
import ru.yandex.mail.cerberus.exception.GroupAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.GroupNotFoundException;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;

import static com.ea.async.Async.await;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.mail.micronaut.common.Async.done;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
class GroupChangeSubject implements ChangeSubject {
    GroupKey group;

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.GROUP;
    }
}

@Value
@Introspected
class GroupAttendeeChange {
    GroupKey group;
    Set<Uid> users;
}

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class GroupEntityInfoProvider implements EntityInfoProvider<GroupKey, GroupEntity> {
    static final GroupEntityInfoProvider INSTANCE = new GroupEntityInfoProvider();

    @Override
    public GroupKey getId(GroupEntity entity) {
        return entity.extractKey();
    }

    @Override
    public SubjectExtractor<GroupEntity> subjectExtractor() {
        return entity -> new GroupChangeSubject(entity.extractKey());
    }

    @Override
    public SubjectExtractor<GroupKey> idSubjectExtractor() {
        return GroupChangeSubject::new;
    }

    @Override
    public RuntimeException notFoundException(Collection<GroupKey> groupKeys) {
        return new GroupNotFoundException(groupKeys);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<GroupKey> groupKeys) {
        return new GroupAlreadyExistsException(groupKeys);
    }
}

@Slf4j
@Singleton
public class DefaultGroupManager implements GroupManager {
    private final GroupRepositoryGroup groupRepositories;
    private final UserRepository userRepository;
    private final GroupMapper mapper;
    private final ChangeLog changeLog;
    private volatile GroupKey defaultGroupKey;
    private final CrudManager<GroupKey, GroupEntity> crudManager;

    @Inject
    public DefaultGroupManager(GroupRepositoryGroup groupRepositories, UserRepository userRepository, ChangeLog changeLog,
                               GroupMapper mapper, TxManagerGroup txManagerGroup) {
        this.groupRepositories = groupRepositories;
        this.userRepository = userRepository;
        this.mapper = mapper;
        this.changeLog = changeLog;
        this.defaultGroupKey = null;
        this.crudManager = new CrudManager<>(groupRepositories.getWriting(), groupRepositories.getReading(),
            txManagerGroup, changeLog, GroupEntityInfoProvider.INSTANCE);
    }

    private static ChangeSubject extractSubject(GroupAttendeeChange change) {
        return new GroupChangeSubject(change.getGroup());
    }

    @Override
    public CompletableFuture<GroupKey> findDefaultGroup() {
        val cachedKey = defaultGroupKey;
        if (cachedKey != null) {
            return completedFuture(cachedKey);
        }

        val repository = groupRepositories.getReading(ReadTarget.SLAVE);
        return crudManager.readingTxManager(ReadTarget.SLAVE).executeAsync(() -> {
            val id = repository.findFirstGroupId(GroupType.INTERNAL, "default")
                .orElseThrow(() -> new IllegalStateException("Default group not found"));

            val key = new GroupKey(id, GroupType.INTERNAL);
            defaultGroupKey = key;
            return key;
        });
    }

    @Override
    public <T> CompletableFuture<Group<T>> insert(Group<T> group) {
        val entity = mapper.mapToEntity(group);
        await(crudManager.insert(entity));
        return done(group);
    }

    @Override
    public <T> CompletableFuture<List<Group<T>>> insert(List<Group<T>> groups) {
        val entities = mapToList(groups, mapper::mapToEntity);
        await(crudManager.insert(CollisionStrategy.FAIL, entities));
        return done(groups);
    }

    @Override
    public <T> CompletableFuture<Group<T>> create(GroupData<T> data) {
        val entity = mapper.mapToEntity(data);
        val insertedEntity = await(crudManager.insert(entity));
        return done(new Group<>(insertedEntity.getId(), data));
    }

    @Override
    public <T> CompletableFuture<List<Group<T>>> create(List<GroupData<T>> data, Class<T> infoType) {
        val entities = mapToList(data, mapper::mapToEntity);
        val insertedEntities = await(crudManager.insert(CollisionStrategy.FAIL, entities));
        return done(mapToList(insertedEntities, entity -> mapper.mapToGroup(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Void> update(Collection<Group<T>> groups) {
        val entities = mapToList(groups, mapper::mapToEntity);
        return crudManager.update(entities);
    }

    @Override
    public <T> CompletableFuture<List<Group<T>>> findGroups(Class<T> infoType, GroupType type, Set<GroupId> ids, ReadTarget readTarget) {
        val entities = crudManager.readingTxManager(readTarget).executeAsync(()-> {
            return groupRepositories.getReading(readTarget)
                .findByType(type, ids);
        });

        return done(mapToList(await(entities), entity -> mapper.mapToGroup(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Page<GroupKey, Group<T>>> groups(Pageable<GroupKey> pageable, Class<T> infoType, ReadTarget readTarget) {
        val page = await(crudManager.findAll(pageable, readTarget));
        return done(page.mapElements(entity -> mapper.mapToGroup(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Page<GroupId, Group<T>>> groups(GroupType type, Pageable<GroupId> pageable, Class<T> infoType,
                                                                 ReadTarget readTarget) {
        val page = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return groupRepositories.getReading(readTarget).findPageByType(pageable, type);
        });
        return done(await(page).mapElements(entity -> mapper.mapToGroup(entity, infoType)));
    }

    @Override
    public CompletableFuture<Void> delete(GroupKey key, DeletionMode mode) {
        return crudManager.deleteById(key, mode);
    }

    @Override
    public void addUserSync(GroupKey groupKey, Uid uid) {
        val oldAttendees = new GroupAttendeeChange(groupKey, emptySet());
        val newAttendees = new GroupAttendeeChange(groupKey, singleton(uid));

        userRepository.addToGroup(uid, groupKey.getId(), groupKey.getType());
        changeLog.writeUpdating(oldAttendees, newAttendees, DefaultGroupManager::extractSubject);
    }

    @Override
    public CompletableFuture<Void> addUser(GroupKey groupKey, Uid uid) {
        return crudManager.writingTxManager()
            .runAsync(() -> addUserSync(groupKey, uid));
    }

    @Override
    public CompletableFuture<Void> removeUser(GroupKey groupKey, Uid uid) {
        val oldAttendees = new GroupAttendeeChange(groupKey, singleton(uid));
        val newAttendees = new GroupAttendeeChange(groupKey, emptySet());

        return crudManager.writingTxManager().runAsync(() -> {
            userRepository.removeFromGroup(uid, groupKey.getId(), groupKey.getType());
            changeLog.writeUpdating(oldAttendees, newAttendees, DefaultGroupManager::extractSubject);
        });
    }

    private enum ChangesOrder {
        NEW_SOURCE,
        OLD_SOURCE
    }

    private void writeChange(GroupType groupType, Map<GroupId, Set<Uid>> uidByGroupId, ChangesOrder order) {
        val source = EntryStream.of(uidByGroupId)
            .mapKeyValue((groupId, uids) -> new GroupAttendeeChange(new GroupKey(groupId, groupType), uids))
            .toImmutableList();

        val derivative = StreamEx.of(source)
            .map(change -> new GroupAttendeeChange(change.getGroup(), emptySet()))
            .toImmutableList();

        val newState = (order == ChangesOrder.NEW_SOURCE) ? source : derivative;
        val oldState = (order == ChangesOrder.NEW_SOURCE) ? derivative : source;

        changeLog.writeUpdating(oldState, newState, DefaultGroupManager::extractSubject);
    }

    @Override
    public void addUsersSync(GroupType groupType, Map<Uid, Set<GroupId>> groupIdsByUid) {
        if (groupIdsByUid.isEmpty()) {
            return;
        }
        val repository = groupRepositories.getReading(ReadTarget.SLAVE);
        Set<GroupId> existingIds = new HashSet<>();
        for (GroupEntity groupEntity : repository.findAll()) {
            existingIds.add(groupEntity.getId());
        }
        Map<Uid, Set<GroupId>> toSync = new HashMap<>();
        for (Map.Entry<Uid, Set<GroupId>> entry : groupIdsByUid.entrySet()) {
            Set<GroupId> existingUsergroupIds = new HashSet<>();
            for (GroupId groupId : entry.getValue()) {
                if (existingIds.contains(groupId)) {
                    existingUsergroupIds.add(groupId);
                } else {
                    log.warn("groupid {} not exists for uid {}", groupId, entry.getKey());
                }
            }
            if (!existingUsergroupIds.isEmpty()) {
                toSync.put(entry.getKey(), existingUsergroupIds);
            }
        }

        crudManager.writingTxManager().run(() -> {
            userRepository.addToGroup(groupType, toSync);

            val uidByGroupId = EntryStream.of(toSync)
                .flatMapValues(StreamEx::of)
                .invert()
                .grouping(Collectors.toSet());

            writeChange(groupType, uidByGroupId, ChangesOrder.NEW_SOURCE);
        });
    }

    @Override
    public CompletableFuture<Void> addUsers(GroupType groupType, Map<Uid, Set<GroupId>> groupIdsByUid) {
        return crudManager.writingTxManager()
            .runAsync(() -> addUsersSync(groupType, groupIdsByUid));
    }

    @Override
    public CompletableFuture<Void> removeUsers(GroupType groupType, Map<Uid, Set<GroupId>> groupIdsByUid) {
        if (groupIdsByUid.isEmpty()) {
            return done();
        }

        return crudManager.writingTxManager().runAsync(() -> {
            val removeResult = userRepository.removeFromGroup(groupType, groupIdsByUid);
            writeChange(groupType, removeResult, ChangesOrder.OLD_SOURCE);
        });
    }
}
