package ru.yandex.mail.cerberus.core.location;

import com.fasterxml.jackson.annotation.JsonCreator;
import io.micronaut.core.annotation.Introspected;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.LocationKey;
import ru.yandex.mail.cerberus.LocationType;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.exception.LocationAlreadyExistsException;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.client.dto.Location;
import ru.yandex.mail.cerberus.client.dto.LocationData;
import ru.yandex.mail.cerberus.core.CrudManager;
import ru.yandex.mail.cerberus.core.EntityInfoProvider;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.core.change_log.ChangeSubject;
import ru.yandex.mail.cerberus.core.change_log.SubjectExtractor;
import ru.yandex.mail.cerberus.exception.LocationNotFoundException;
import ru.yandex.mail.cerberus.core.mapper.LocationMapper;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.location.LocationEntity;
import ru.yandex.mail.cerberus.dao.location.LocationRepositoryGroup;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import static com.ea.async.Async.await;
import static ru.yandex.mail.micronaut.common.Async.done;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
class LocationChangeSubject implements ChangeSubject {
    LocationKey location;

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.LOCATION;
    }
}

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class LocationEntityInfoProvider implements EntityInfoProvider<LocationKey, LocationEntity> {
    static final LocationEntityInfoProvider INSTANCE = new LocationEntityInfoProvider();

    @Override
    public LocationKey getId(LocationEntity locationEntity) {
        return locationEntity.extractKey();
    }

    @Override
    public SubjectExtractor<LocationEntity> subjectExtractor() {
        return entity -> new LocationChangeSubject(entity.extractKey());
    }

    @Override
    public SubjectExtractor<LocationKey> idSubjectExtractor() {
        return LocationChangeSubject::new;
    }

    @Override
    public RuntimeException notFoundException(Collection<LocationKey> locationKeys) {
        return new LocationNotFoundException(locationKeys);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<LocationKey> locationKeys) {
        return new LocationAlreadyExistsException(locationKeys);
    }
}

@Slf4j
@Singleton
public class DefaultLocationManager implements LocationManager {
    private final LocationRepositoryGroup locationRepositories;
    private final LocationMapper mapper;
    private final CrudManager<LocationKey, LocationEntity> crudManager;

    @Inject
    public DefaultLocationManager(LocationRepositoryGroup locationRepositories, LocationMapper mapper, ChangeLog changeLog,
                                  TxManagerGroup txManagerGroup) {
        this.locationRepositories = locationRepositories;
        this.mapper = mapper;
        crudManager = new CrudManager<>(locationRepositories.getWriting(), locationRepositories.getReading(), txManagerGroup,
            changeLog, LocationEntityInfoProvider.INSTANCE);
    }

    @Override
    public <T> CompletableFuture<Location<T>> create(LocationData<T> data) {
        val entity = mapper.mapToEntity(data);
        val insertedEntity = await(crudManager.insert(entity));
        return done(new Location<>(insertedEntity.getId(), data));
    }

    @Override
    public <T> CompletableFuture<List<Location<T>>> create(List<LocationData<T>> locationData, Class<T> infoType) {
        val entities = mapToList(locationData, mapper::mapToEntity);
        val insertedEntities = await(crudManager.insert(CollisionStrategy.FAIL, entities));
        return done(mapToList(insertedEntities, entity -> mapper.mapToLocation(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Location<T>> insert(Location<T> location) {
        val entity = mapper.mapToEntity(location);
        await(crudManager.insert(entity));
        return done(location);
    }

    @Override
    public <T> CompletableFuture<List<Location<T>>> insert(List<Location<T>> locations) {
        val entities = mapToList(locations, mapper::mapToEntity);
        await(crudManager.insert(CollisionStrategy.FAIL, entities));
        return done(locations);
    }

    @Override
    public <T> CompletableFuture<Void> update(List<Location<T>> locations) {
        val entities = mapToList(locations, mapper::mapToEntity);
        return crudManager.update(entities);
    }

    @Override
    public CompletableFuture<Void> delete(LocationKey key, DeletionMode mode) {
        return crudManager.deleteById(key, mode);
    }

    @Override
    public <T> CompletableFuture<Page<LocationKey, Location<T>>> locations(Pageable<LocationKey> pageable, Class<T> infoClass,
                                                                           ReadTarget readTarget) {
        val page = await(crudManager.findAll(pageable, readTarget));
        return done(page.mapElements(entity -> mapper.mapToLocation(entity, infoClass)));
    }

    @Override
    public <T> CompletableFuture<Page<LocationId, Location<T>>> locationsByType(LocationType type, Pageable<LocationId> pageable,
                                                                                Class<T> infoClass, ReadTarget readTarget) {
        val repository = locationRepositories.getReading(readTarget);
        val page = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return repository.findPageByType(pageable, type);
        });
        return done(await(page).mapElements(entity -> mapper.mapToLocation(entity, infoClass)));
    }

    @Override
    public <T> CompletableFuture<List<Location<T>>> findByType(Class<T> infoClass, LocationType type, Set<LocationId> ids,
                                                               ReadTarget readTarget) {
        val repository = locationRepositories.getReading(readTarget);
        val entities = await(crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return repository.findByType(type, ids);
        }));
        return done(mapToList(entities, entity -> mapper.mapToLocation(entity, infoClass)));
    }
}
