package ru.yandex.mail.cerberus.core.resource;

import com.fasterxml.jackson.annotation.JsonCreator;
import io.micronaut.core.annotation.Introspected;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.Value;
import lombok.val;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.exception.ResourceAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.ResourceTypeAlreadyExistsException;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceKey;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.client.dto.Resource;
import ru.yandex.mail.cerberus.client.dto.ResourceData;
import ru.yandex.mail.cerberus.client.dto.ResourceType;
import ru.yandex.mail.cerberus.core.CrudManager;
import ru.yandex.mail.cerberus.core.EntityInfoProvider;
import ru.yandex.mail.cerberus.core.change_log.ChangeSubject;
import ru.yandex.mail.cerberus.core.change_log.SubjectExtractor;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.exception.ResourceNotFoundException;
import ru.yandex.mail.cerberus.exception.ResourceTypeNotFoundException;
import ru.yandex.mail.cerberus.core.mapper.ResourceMapper;
import ru.yandex.mail.cerberus.core.mapper.ResourceTypeMapper;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.resource.ResourceEntity;
import ru.yandex.mail.cerberus.dao.resource.ResourceRepository;
import ru.yandex.mail.cerberus.dao.resource.ResourceRepositoryGroup;
import ru.yandex.mail.cerberus.dao.resource_type.ResourceTypeEntity;
import ru.yandex.mail.cerberus.dao.resource_type.ResourceTypeRepository;
import ru.yandex.mail.cerberus.dao.resource_type.ResourceTypeRepositoryGroup;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import static com.ea.async.Async.await;
import static ru.yandex.mail.micronaut.common.Async.done;
import static ru.yandex.mail.micronaut.common.CerberusUtils.findMissing;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
class ResourceTypeSubject implements ChangeSubject {
    ResourceTypeName type;

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.RESOURCE_TYPE;
    }
}

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
class ResourceChangeSubject implements ChangeSubject {
    ResourceKey resource;

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.RESOURCE;
    }
}

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class ResourceTypeEntityInfoProvider implements EntityInfoProvider<ResourceTypeName, ResourceTypeEntity> {
    static final ResourceTypeEntityInfoProvider INSTANCE = new ResourceTypeEntityInfoProvider();

    @Override
    public ResourceTypeName getId(ResourceTypeEntity entity) {
        return entity.getName();
    }

    @Override
    public SubjectExtractor<ResourceTypeEntity> subjectExtractor() {
        return entity -> new ResourceTypeSubject(entity.getName());
    }

    @Override
    public SubjectExtractor<ResourceTypeName> idSubjectExtractor() {
        return ResourceTypeSubject::new;
    }

    @Override
    public RuntimeException notFoundException(Collection<ResourceTypeName> resourceTypeNames) {
        return new ResourceTypeNotFoundException(resourceTypeNames);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<ResourceTypeName> resourceTypeNames) {
        return new ResourceTypeAlreadyExistsException(resourceTypeNames);
    }
}

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class ResourceEntityInfoProvider implements EntityInfoProvider<ResourceKey, ResourceEntity> {
    static final ResourceEntityInfoProvider INSTANCE = new ResourceEntityInfoProvider();

    @Override
    public ResourceKey getId(ResourceEntity entity) {
        return entity.extractKey();
    }

    @Override
    public SubjectExtractor<ResourceEntity> subjectExtractor() {
        return entity -> new ResourceChangeSubject(entity.extractKey());
    }

    @Override
    public SubjectExtractor<ResourceKey> idSubjectExtractor() {
        return ResourceChangeSubject::new;
    }

    @Override
    public RuntimeException notFoundException(Collection<ResourceKey> resourceKeys) {
        return new ResourceNotFoundException(resourceKeys);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<ResourceKey> resourceKeys) {
        return new ResourceAlreadyExistsException(resourceKeys);
    }
}

@Singleton
public class DefaultResourceManager implements ResourceManager {
    private final ResourceTypeRepositoryGroup resourceTypeRepositories;
    private final ResourceRepositoryGroup resourceRepositories;
    private final ResourceTypeMapper typeMapper;
    private final ResourceMapper mapper;
    private final ChangeLog changeLog;
    private final CrudManager<ResourceTypeName, ResourceTypeEntity> typeCrudManager;
    private final CrudManager<ResourceKey, ResourceEntity> crudManager;

    @Inject
    public DefaultResourceManager(ResourceTypeRepositoryGroup resourceTypeRepositories, ResourceRepositoryGroup resourceRepositories,
                                  ResourceTypeMapper typeMapper, ResourceMapper mapper, ChangeLog changeLog, TxManagerGroup txManagerGroup) {
        this.resourceTypeRepositories = resourceTypeRepositories;
        this.resourceRepositories = resourceRepositories;
        this.typeMapper = typeMapper;
        this.mapper = mapper;
        this.changeLog = changeLog;
        this.typeCrudManager = new CrudManager<>(resourceTypeRepositories.getWriting(), resourceTypeRepositories.getReading(),
            txManagerGroup, changeLog, ResourceTypeEntityInfoProvider.INSTANCE);
        this.crudManager = new CrudManager<>(resourceRepositories.getWriting(), resourceRepositories.getReading(),
            txManagerGroup, changeLog, ResourceEntityInfoProvider.INSTANCE);
    }

    private ResourceTypeRepository writingTypeRepository() {
        return resourceTypeRepositories.getWriting();
    }

    private ResourceRepository writingResourceRepository() {
        return resourceRepositories.getWriting();
    }

    @Override
    public CompletableFuture<ResourceType> createType(ResourceType data) {
        val entity = typeMapper.mapToEntity(data);
        val insertedEntity = await(typeCrudManager.insert(entity));
        return done(typeMapper.mapToType(insertedEntity));
    }

    @Override
    public CompletableFuture<ResourceType> getByNameOrCreateType(ResourceType data) {
        val entity = typeMapper.mapToEntity(data);
        return typeCrudManager.writingTxManager().executeAsync(() -> {
            val result = writingTypeRepository().createIfNotExist(entity);
            val newEntity = result.getEntity();
            if (result.isInserted()) {
                changeLog.writeCreation(newEntity, ResourceTypeEntityInfoProvider.INSTANCE.subjectExtractor());
            }
            return data;
        });
    }

    @Override
    public CompletableFuture<List<ResourceType>> getTypes(Set<ResourceTypeName> names, ReadTarget readTarget) {
        val entities = typeCrudManager.readingTxManager(readTarget).executeAsync(() -> {
            return resourceTypeRepositories.getReading(readTarget).findAll(names);
        });
        return done(mapToList(await(entities), typeMapper::mapToType));
    }

    @Override
    public CompletableFuture<Set<String>> getPossibleActions(ResourceTypeName type, ReadTarget readTarget) {
        return typeCrudManager.readingTxManager(readTarget).executeAsync(() -> {
            return resourceTypeRepositories.getReading(readTarget)
                .find(type)
                .orElseThrow(() -> new ResourceTypeNotFoundException(type))
                .getActionSet();
        });
    }

    @Override
    public <T> CompletableFuture<Resource<T>> createResource(ResourceData<T> resourceData) {
        val entity = mapper.mapToEntity(resourceData);
        val insertedEntity = await(crudManager.insert(entity));
        return done(new Resource<>(insertedEntity.getId(), resourceData));
    }

    @Override
    public <T> CompletableFuture<List<Resource<T>>> createResources(List<ResourceData<T>> resourcesData, Class<T> infoType) {
        val entities = mapToList(resourcesData, mapper::mapToEntity);
        val insertedEntities = await(crudManager.insert(CollisionStrategy.FAIL, entities));
        return done(mapToList(insertedEntities, entity -> mapper.mapToResource(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<List<Resource<T>>> insertResources(CollisionStrategy collisionStrategy, List<Resource<T>> resources) {
        val entities = mapToList(resources, mapper::mapToEntity);
        await(crudManager.insert(collisionStrategy, entities));
        return done(resources);
    }

    @Override
    public <T> CompletableFuture<Page<ResourceKey, Resource<T>>> resources(Pageable<ResourceKey> pageable, Class<T> infoType,
                                                                           ReadTarget readTarget) {
        val page = await(crudManager.findAll(pageable, readTarget));
        return done(page.mapElements(entity -> mapper.mapToResource(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Page<ResourceId, Resource<T>>> resources(ResourceTypeName type, Pageable<ResourceId> pageable,
                                                                          Class<T> infoType, ReadTarget readTarget) {
        val repository = resourceRepositories.getReading(readTarget);
        val page = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return repository.findPageByType(pageable, type);
        });
        return done(await(page).mapElements(entity -> mapper.mapToResource(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<List<Resource<T>>> findResources(ResourceTypeName type, Set<ResourceId> ids, Class<T> infoType,
                                                                  ReadTarget readTarget) {
        val entities = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return resourceRepositories.getReading(readTarget).findById(type, ids);
        });

        return done(mapToList(await(entities), entity -> mapper.mapToResource(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Void> updateResource(Resource<T> resource) {
        val entity = mapper.mapToEntity(resource);
        return crudManager.update(entity);
    }

    @Override
    public <T> CompletableFuture<Void> updateResources(Collection<Resource<T>> resources) {
        val entities = mapToList(resources, mapper::mapToEntity);
        return crudManager.update(entities);
    }

    @Override
    public CompletableFuture<Void> deleteResource(ResourceKey key, DeletionMode mode) {
        return crudManager.deleteById(key, mode);
    }

    @Override
    public CompletableFuture<Void> deleteResources(ResourceTypeName type, Set<ResourceId> ids, DeletionMode mode) {
        if (ids.isEmpty()) {
            return done();
        }

        return crudManager.writingTxManager().runAsync(() -> {
            val deletedKeys = writingResourceRepository().deleteAll(type, ids);

            if (mode == DeletionMode.STRICT && deletedKeys.size() != ids.size()) {
                val missingIds = findMissing(ids, mapToList(deletedKeys, ResourceKey::getId));
                throw new ResourceNotFoundException(mapToList(missingIds, id -> new ResourceKey(id, type)));
            }

            changeLog.writeDeletion(deletedKeys, ResourceChangeSubject::new);
        });
    }
}
