package ru.yandex.mail.cerberus.core.role;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.micronaut.core.annotation.Introspected;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.Value;
import lombok.val;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.exception.RoleAlreadyExistsException;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.Role;
import ru.yandex.mail.cerberus.client.dto.RoleData;
import ru.yandex.mail.cerberus.core.CrudManager;
import ru.yandex.mail.cerberus.core.EntityInfoProvider;
import ru.yandex.mail.cerberus.core.change_log.ChangeSubject;
import ru.yandex.mail.cerberus.core.change_log.LongIdSubject;
import ru.yandex.mail.cerberus.core.change_log.SubjectExtractor;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.exception.RoleNotFoundException;
import ru.yandex.mail.cerberus.core.mapper.RoleMapper;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.group.GroupRepository;
import ru.yandex.mail.cerberus.dao.role.RoleEntity;
import ru.yandex.mail.cerberus.dao.role.RoleRepository;
import ru.yandex.mail.cerberus.dao.role.RoleRepositoryGroup;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;
import ru.yandex.mail.cerberus.dao.user.UserRepository;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import static com.ea.async.Async.await;
import static ru.yandex.mail.micronaut.common.Async.done;

@Value
@Introspected
@JsonInclude(JsonInclude.Include.NON_ABSENT)
@AllArgsConstructor(onConstructor_= @JsonCreator)
class RolesChange {
    Optional<Uid> user;
    Optional<GroupKey> group;
    Set<RoleId> roles;

    RolesChange(Uid uid, Set<RoleId> roles) {
        this(Optional.of(uid), Optional.empty(), roles);
    }

    RolesChange(GroupKey groupKey, Set<RoleId> roles) {
        this(Optional.empty(), Optional.of(groupKey), roles);
    }
}

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
class GroupChangeSubject implements ChangeSubject {
    GroupKey group;

    @Override
    public ChangeSubjectType changeType() {
        return ChangeSubjectType.GROUP;
    }
}

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class RoleEntityInfoProvider implements EntityInfoProvider<RoleId, RoleEntity> {
    static final RoleEntityInfoProvider INSTANCE = new RoleEntityInfoProvider();

    @Override
    public RoleId getId(RoleEntity entity) {
        return entity.getId();
    }

    @Override
    public SubjectExtractor<RoleEntity> subjectExtractor() {
        return entity -> new LongIdSubject(entity.getId(), ChangeSubjectType.ROLE);
    }

    @Override
    public SubjectExtractor<RoleId> idSubjectExtractor() {
        return id -> new LongIdSubject(id, ChangeSubjectType.ROLE);
    }

    @Override
    public RuntimeException notFoundException(Collection<RoleId> roleIds) {
        return new RoleNotFoundException(roleIds);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<RoleId> roleIds) {
        return new RoleAlreadyExistsException(roleIds);
    }
}

@Singleton
public class DefaultRoleManager implements RoleManager {
    private final RoleMapper mapper;
    private final RoleRepositoryGroup roleRepositories;
    private final GroupRepository groupRepository;
    private final UserRepository userRepository;
    private final ChangeLog changeLog;
    private final CrudManager<RoleId, RoleEntity> crudManager;

    @Inject
    public DefaultRoleManager(RoleMapper mapper, RoleRepositoryGroup roleRepositories, GroupRepository groupRepository, UserRepository userRepository, ChangeLog changeLog, TxManagerGroup txManagerGroup) {
        this.mapper = mapper;
        this.roleRepositories = roleRepositories;
        this.groupRepository = groupRepository;
        this.userRepository = userRepository;
        this.changeLog = changeLog;
        this.crudManager = new CrudManager<>(roleRepositories.getWriting(), roleRepositories.getReading(), txManagerGroup,
            changeLog, RoleEntityInfoProvider.INSTANCE);
    }

    private RoleRepository writingRepository() {
        return roleRepositories.getWriting();
    }

    private static ChangeSubject extractSubject(RolesChange change) {
        return change.getUser()
            .map(uid -> new LongIdSubject(uid, ChangeSubjectType.USER))
            .map(ChangeSubject.class::cast)
            .or(() -> change.getGroup().map(GroupChangeSubject::new))
            .get();
    }

    @Override
    public CompletableFuture<Role> create(RoleData data) {
        val entity = mapper.mapToEntity(data);
        val insertedEntity = await(crudManager.insert(entity));
        return done(mapper.mapToRole(insertedEntity));
    }

    @Override
    public CompletableFuture<Void> update(Role role) {
        val entity = mapper.mapToEntity(role);
        return crudManager.update(entity);
    }

    @Override
    public CompletableFuture<Void> delete(RoleId id, DeletionMode mode) {
        return crudManager.deleteById(id, mode);
    }

    @Override
    public CompletableFuture<Void> delete(String name, DeletionMode mode) {
        return crudManager.writingTxManager().runAsync(() -> {
            val deletedId = writingRepository().deleteByName(name);
            if (mode == DeletionMode.STRICT && deletedId.isEmpty()) {
                throw new RoleNotFoundException(name);
            }

            deletedId.ifPresent(id -> {
                changeLog.writeDeletion(new LongIdSubject(id, ChangeSubjectType.ROLE));
            });
        });
    }

    @Override
    public CompletableFuture<Page<RoleId, Role>> roles(Pageable<RoleId> pageable, ReadTarget readTarget) {
        val page = await(crudManager.findAll(pageable, readTarget));
        return done(page.mapElements(mapper::mapToRole));
    }

    @Override
    public CompletableFuture<Boolean> exists(RoleId id, ReadTarget readTarget) {
        val repository = roleRepositories.getReading(readTarget);
        return crudManager.readingTxManager(readTarget).executeAsync(() -> repository.exists(id));
    }

    private CompletableFuture<Void> attachRole(Uid uid, RoleId roleId, Runnable inserter) {
        val oldRoles = new RolesChange(uid, Set.of());
        val newRoles = new RolesChange(uid, Set.of(roleId));
        return attachRole(oldRoles, newRoles, roleId, inserter);
    }

    private CompletableFuture<Void> attachRole(GroupKey groupKey, RoleId roleId, Runnable inserter) {
        val oldRoles = new RolesChange(groupKey, Set.of());
        val newRoles = new RolesChange(groupKey, Set.of(roleId));
        return attachRole(oldRoles, newRoles, roleId, inserter);
    }

    private CompletableFuture<Void> attachRole(RolesChange oldRoles, RolesChange newRoles, RoleId roleId, Runnable inserter) {
        return crudManager.writingTxManager().runAsync(() -> {
            if (!writingRepository().exists(roleId)) {
                throw new RoleNotFoundException(roleId);
            }

            inserter.run();
            changeLog.writeUpdating(oldRoles, newRoles, DefaultRoleManager::extractSubject);
        });
    }

    private CompletableFuture<Void> detachRole(Uid uid, RoleId roleId, Runnable deleter) {
        val oldRoles = new RolesChange(uid, Set.of(roleId));
        val newRoles = new RolesChange(uid, Set.of());
        return detachRole(oldRoles, newRoles, roleId, deleter);
    }

    private CompletableFuture<Void> detachRole(GroupKey groupKey, RoleId roleId, Runnable deleter) {
        val oldRoles = new RolesChange(groupKey, Set.of(roleId));
        val newRoles = new RolesChange(groupKey, Set.of());
        return detachRole(oldRoles, newRoles, roleId, deleter);
    }

    private CompletableFuture<Void> detachRole(RolesChange oldRoles, RolesChange newRoles, RoleId roleId, Runnable deleter) {
        return crudManager.writingTxManager().runAsync(() -> {
            if (!writingRepository().exists(roleId)) {
                throw new RoleNotFoundException(roleId);
            }

            deleter.run();
            changeLog.writeUpdating(oldRoles, newRoles, DefaultRoleManager::extractSubject);
        });
    }

    @Override
    public CompletableFuture<Void> attachRoleToGroup(GroupKey groupKey, RoleId roleId) {
        return attachRole(groupKey, roleId, () -> {
            groupRepository.attachRole(groupKey.getId(), groupKey.getType(), roleId);
        });
    }

    @Override
    public CompletableFuture<Void> attachRoleToUser(Uid uid, RoleId roleId) {
        return attachRole(uid, roleId, () -> {
            userRepository.attachRole(uid, roleId);
        });
    }

    @Override
    public CompletableFuture<Void> detachRoleFromGroup(GroupKey groupKey, RoleId roleId) {
        return detachRole(groupKey, roleId, () -> {
            groupRepository.detachRole(groupKey.getId(), groupKey.getType(), roleId);
        });
    }

    @Override
    public CompletableFuture<Void> detachRoleFromUser(Uid uid, RoleId roleId) {
        return detachRole(uid, roleId, () -> {
            userRepository.detachRole(uid, roleId);
        });
    }
}
