package ru.yandex.mail.cerberus.core.user;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.client.dto.UserWithGroups;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.exception.UserAlreadyExistsException;
import ru.yandex.mail.cerberus.core.group.GroupManager;
import ru.yandex.mail.cerberus.dao.user.RoUserRepository;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.cerberus.core.CrudManager;
import ru.yandex.mail.cerberus.core.EntityInfoProvider;
import ru.yandex.mail.cerberus.core.change_log.LongIdSubject;
import ru.yandex.mail.cerberus.core.change_log.SubjectExtractor;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.change_log.ChangeLog;
import ru.yandex.mail.cerberus.exception.UserNotFoundException;
import ru.yandex.mail.cerberus.core.mapper.UserMapper;
import ru.yandex.mail.cerberus.dao.change_log.ChangeSubjectType;
import ru.yandex.mail.cerberus.dao.tx.TxManagerGroup;
import ru.yandex.mail.cerberus.dao.user.UserEntity;
import ru.yandex.mail.cerberus.dao.user.UserRepositoryGroup;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import static com.ea.async.Async.await;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.mail.micronaut.common.Async.done;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToMap;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
class UserEntityInfoProvider implements EntityInfoProvider<Uid, UserEntity> {
    static final UserEntityInfoProvider INSTANCE = new UserEntityInfoProvider();

    @Override
    public Uid getId(UserEntity entity) {
        return entity.getUid();
    }

    @Override
    public SubjectExtractor<UserEntity> subjectExtractor() {
        return entity -> new LongIdSubject(entity.getUid(), ChangeSubjectType.USER);
    }

    @Override
    public SubjectExtractor<Uid> idSubjectExtractor() {
        return uid -> new LongIdSubject(uid, ChangeSubjectType.USER);
    }

    @Override
    public RuntimeException notFoundException(Collection<Uid> uids) {
        return new UserNotFoundException(uids);
    }

    @Override
    public RuntimeException alreadyExistsException(Collection<Uid> uids) {
        return new UserAlreadyExistsException(uids);
    }
}

@Singleton
public class DefaultUserManager implements UserManager {
    private final UserRepositoryGroup userRepositories;
    private final UserMapper mapper;
    private final TxManagerGroup txManagerGroup;
    private final CrudManager<Uid, UserEntity> crudManager;
    private final GroupManager groupManager;

    @Inject
    public DefaultUserManager(UserRepositoryGroup userRepositories, UserMapper mapper, ChangeLog changeLog,
                              TxManagerGroup txManagerGroup, GroupManager groupManager) {
        this.userRepositories = userRepositories;
        this.mapper = mapper;
        this.txManagerGroup = txManagerGroup;
        this.groupManager = groupManager;
        this.crudManager = new CrudManager<>(userRepositories.getWriting(), userRepositories.getReading(), txManagerGroup,
            changeLog, UserEntityInfoProvider.INSTANCE);
    }

    @Override
    public <T> CompletableFuture<User<T>> insert(User<T> user) {
        return insert(CollisionStrategy.FAIL, singletonList(user))
            .thenApply(list -> list.get(0));
    }

    @Override
    public <T> CompletableFuture<List<User<T>>> insert(CollisionStrategy collisionStrategy, List<User<T>> users) {
        val entities = mapToList(users, mapper::mapToEntity);
        val defaultGroupId = await(groupManager.findDefaultGroup()).getId();
        final var mapping = mapToMap(users, User::getUid, ignored -> singleton(defaultGroupId));

        return crudManager.writingTxManager().executeAsync(() -> {
            crudManager.insertSync(collisionStrategy, entities);
            groupManager.addUsersSync(GroupType.INTERNAL, mapping);
            return users;
        });
    }

    @Override
    public <T> CompletableFuture<Void> update(User<T> user) {
        val entity = mapper.mapToEntity(user);
        return crudManager.update(entity);
    }

    @Override
    public <T> CompletableFuture<Void> update(Collection<User<T>> users) {
        val entities = mapToList(users, mapper::mapToEntity);
        return crudManager.update(entities);
    }

    @Override
    public <T> CompletableFuture<Optional<User<T>>> find(Uid uid, Class<T> infoType, ReadTarget readTarget) {
        val entity = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return userRepositories.getReading(readTarget).find(uid);
        });
        return completedFuture(await(entity).map(e -> mapper.mapToUser(e, infoType)));
    }

    @Override
    public <T> CompletableFuture<List<User<T>>> find(Collection<Uid> uids, Class<T> infoType, ReadTarget readTarget) {
        val entities = crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return userRepositories.getReading(readTarget).findAll(uids);
        });
        return done(mapToList(await(entities), e -> mapper.mapToUser(e, infoType)));
    }

    @Override
    public CompletableFuture<Map<Uid, Set<GroupId>>> findUsersGroupsByType(Collection<Uid> uids, GroupType groupType,
                                                                           ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            return userRepositories
                .getReading(readTarget)
                .findUserGroupsByType(uids, groupType)
                .getSetMapping();
        });
    }

    @Override
    public <T> CompletableFuture<Page<Uid, User<T>>> users(Pageable<Uid> pageable, Class<T> infoType, ReadTarget readTarget) {
        val page = await(crudManager.findAll(pageable, readTarget));
        return done(page.mapElements(entity -> mapper.mapToUser(entity, infoType)));
    }

    @Override
    public <T> CompletableFuture<Page<Uid, User<T>>> users(UserType type, Pageable<Uid> pageable, Class<T> infoType,
                                                           ReadTarget readTarget) {
        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            return userRepositories
                .getReading(readTarget)
                .findPageByType(pageable, type)
                .mapElements(entity -> mapper.mapToUser(entity, infoType));
        });
    }

    private <T> Page<Uid, UserWithGroups<T>> usersWithGroups(Page<Uid, User<T>> page, RoUserRepository repository) {
        val uids = StreamEx.of(page.getElements())
            .map(User::getUid)
            .toImmutableList();

        val usersGroupKeys = repository.findUsersGroupKeys(uids)
            .getSetMapping();

        return page.mapElements(user -> {
            val groupKeys = requireNonNull(usersGroupKeys.get(user.getUid()));
            return new UserWithGroups<>(user, groupKeys);
        });
    }

    @Override
    public <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(UserType type, Pageable<Uid> pageable,
                                                                               Class<T> infoType, ReadTarget readTarget) {
        val repository = userRepositories.getReading(readTarget);

        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            val page = repository
                .findPageByType(pageable, type)
                .mapElements(entity -> mapper.mapToUser(entity, infoType));
            return usersWithGroups(page, repository);
        });
    }

    @Override
    public <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(Pageable<Uid> pageable, Class<T> infoType,
                                                                               ReadTarget readTarget) {
        val repository = userRepositories.getReading(readTarget);

        return txManagerGroup.getReading(readTarget).executeAsync(() -> {
            val page = repository
                .findPage(pageable)
                .mapElements(entity -> mapper.mapToUser(entity, infoType));
            return usersWithGroups(page, repository);
        });
    }

    @Override
    public CompletableFuture<Boolean> exist(Uid uid, ReadTarget readTarget) {
        return crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return userRepositories.getReading(readTarget).exists(uid);
        });
    }

    @Override
    public CompletableFuture<Boolean> isSuperuser(Uid uid, ReadTarget readTarget) {
        return crudManager.readingTxManager(readTarget).executeAsync(() -> {
            return userRepositories.getReading(readTarget)
                .isSuperuser(uid)
                .orElse(false);
        });
    }

    @Override
    public CompletableFuture<Void> delete(Uid uid, DeletionMode mode) {
        return crudManager.deleteById(uid, mode);
    }

    @Override
    public CompletableFuture<Void> deleteByUid(Collection<Uid> uids, DeletionMode mode) {
        return crudManager.deleteById(uids, mode);
    }
}
