package ru.yandex.mail.cerberus.core.user;

import io.micronaut.context.annotation.DefaultImplementation;
import one.util.streamex.StreamEx;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.client.dto.UserWithGroups;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.cerberus.ReadTarget;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

@DefaultImplementation(DefaultUserManager.class)
public interface UserManager {
    default ReadTarget defaultReadTarget() {
        return ReadTarget.SLAVE;
    }

    <T> CompletableFuture<User<T>> insert(User<T> user);

    <T> CompletableFuture<List<User<T>>> insert(CollisionStrategy collisionStrategy, List<User<T>> users);

    <T> CompletableFuture<Void> update(User<T> user);

    <T> CompletableFuture<Void> update(Collection<User<T>> users);

    <T> CompletableFuture<Optional<User<T>>> find(Uid uid, Class<T> infoType, ReadTarget readTarget);

    default <T> CompletableFuture<Optional<User<T>>> find(Uid uid, Class<T> infoType) {
        return find(uid, infoType, defaultReadTarget());
    }

    <T> CompletableFuture<List<User<T>>> find(Collection<Uid> uids, Class<T> infoType, ReadTarget readTarget);

    default <T> CompletableFuture<List<User<T>>> find(Collection<Uid> uids, Class<T> infoType) {
        return find(uids, infoType, defaultReadTarget());
    }

    CompletableFuture<Map<Uid, Set<GroupId>>> findUsersGroupsByType(Collection<Uid> uids, GroupType groupType, ReadTarget readTarget);

    default CompletableFuture<Map<Uid, Set<GroupId>>> findUsersGroupsByType(Collection<Uid> uids, GroupType groupType) {
        return findUsersGroupsByType(uids, groupType, defaultReadTarget());
    }

    <T> CompletableFuture<Page<Uid, User<T>>> users(Pageable<Uid> pageable, Class<T> infoType, ReadTarget readTarget);

    default <T> CompletableFuture<Page<Uid, User<T>>> users(Pageable<Uid> pageable, Class<T> infoType) {
        return users(pageable, infoType, defaultReadTarget());
    }

    <T> CompletableFuture<Page<Uid, User<T>>> users(UserType type, Pageable<Uid> pageable, Class<T> infoType, ReadTarget readTarget);

    default <T> CompletableFuture<Page<Uid, User<T>>> users(UserType type, Pageable<Uid> pageable, Class<T> infoType) {
        return users(type, pageable, infoType, defaultReadTarget());
    }

    <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(UserType type, Pageable<Uid> pageable,
                                                                        Class<T> infoType, ReadTarget readTarget);

    default <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(UserType type, Pageable<Uid> pageable,
                                                                                Class<T> infoType) {
        return usersWithGroups(type, pageable, infoType, defaultReadTarget());
    }

    <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(Pageable<Uid> pageable, Class<T> infoType,
                                                                        ReadTarget readTarget);

    default <T> CompletableFuture<Page<Uid, UserWithGroups<T>>> usersWithGroups(Pageable<Uid> pageable, Class<T> infoType) {
        return usersWithGroups(pageable, infoType, defaultReadTarget());
    }

    CompletableFuture<Boolean> exist(Uid uid, ReadTarget readTarget);

    default CompletableFuture<Boolean> exist(Uid uid) {
        return exist(uid, defaultReadTarget());
    }

    CompletableFuture<Boolean> isSuperuser(Uid uid, ReadTarget readTarget);

    default CompletableFuture<Boolean> isSuperuser(Uid uid) {
        return isSuperuser(uid, defaultReadTarget());
    }

    CompletableFuture<Void> delete(Uid uid, DeletionMode mode);

    CompletableFuture<Void> deleteByUid(Collection<Uid> uids, DeletionMode mode);

    default <T> CompletableFuture<Void> delete(User<T> user, DeletionMode mode) {
        return delete(user.getUid(), mode);
    }

    default <T> CompletableFuture<Void> delete(Collection<User<T>> users, DeletionMode mode) {
        return deleteByUid(StreamEx.of(users).map(User::getUid).toImmutableList(), mode);
    }
}
