package ru.yandex.mail.cerberus.dao;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.micronaut.context.BeanLocator;
import io.micronaut.context.annotation.Bean;
import io.micronaut.context.annotation.Factory;
import io.micronaut.inject.qualifiers.Qualifiers;
import lombok.val;
import org.jdbi.v3.core.statement.SqlLogger;
import ru.yandex.mail.cerberus.asyncdb.Repository;
import ru.yandex.mail.cerberus.asyncdb.RepositoryFactory;
import ru.yandex.mail.cerberus.dao.change_log.ChangeLogRepository;
import ru.yandex.mail.cerberus.dao.change_log.RoChangeLogRepository;
import ru.yandex.mail.cerberus.dao.general.GeneralRepository;
import ru.yandex.mail.cerberus.dao.grant.GrantRepository;
import ru.yandex.mail.cerberus.dao.grant.RoGrantRepository;
import ru.yandex.mail.cerberus.dao.group.GroupRepository;
import ru.yandex.mail.cerberus.dao.group.RoGroupRepository;
import ru.yandex.mail.cerberus.dao.location.LocationRepository;
import ru.yandex.mail.cerberus.dao.location.RoLocationRepository;
import ru.yandex.mail.cerberus.dao.resource.ResourceRepository;
import ru.yandex.mail.cerberus.dao.resource.RoResourceRepository;
import ru.yandex.mail.cerberus.dao.resource_type.ResourceTypeRepository;
import ru.yandex.mail.cerberus.dao.resource_type.RoResourceTypeRepository;
import ru.yandex.mail.cerberus.dao.role.RoRoleRepository;
import ru.yandex.mail.cerberus.dao.role.RoleRepository;
import ru.yandex.mail.cerberus.dao.task.TaskRepository;
import ru.yandex.mail.cerberus.dao.user.RoUserRepository;
import ru.yandex.mail.cerberus.dao.user.UserRepository;
import ru.yandex.mail.micronaut.common.qualifier.Master;
import ru.yandex.mail.micronaut.common.qualifier.Slave;

import javax.inject.Inject;
import javax.inject.Singleton;
import javax.sql.DataSource;
import java.util.Optional;

import static ru.yandex.mail.cerberus.dao.DaoConstants.MASTER;
import static ru.yandex.mail.cerberus.dao.DaoConstants.SLAVE;

@Factory
public class DaoRepositoriesFactory {
    private final RepositoryFactory masterFactory;
    private final Optional<RepositoryFactory> slaveFactory;

    private static Optional<RepositoryFactory> createFactory(String dataSourceName, BeanLocator beanLocator,
                                                             ObjectMapper objectMapper, Optional<SqlLogger> sqlLogger) {
        return beanLocator.findBean(DataSource.class, Qualifiers.byName(dataSourceName))
            .map(dataSource -> {
                val extConfig = beanLocator.findBean(ExtendedDatasourceConfiguration.class, Qualifiers.byName(dataSourceName));
                val defaultQueryTimeout = extConfig.map(ExtendedDatasourceConfiguration::getDefaultQueryTimeout);
                return new RepositoryFactory(dataSource, objectMapper, defaultQueryTimeout, sqlLogger);
            });
    }

    @Inject
    public DaoRepositoriesFactory(BeanLocator beanLocator, ObjectMapper objectMapper, @Master Optional<SqlLogger> masterSqlLogger,
                                  @Slave Optional<SqlLogger> slaveSqlLogger) {
        masterFactory = createFactory(MASTER, beanLocator, objectMapper, masterSqlLogger)
            .orElseThrow(() -> new RuntimeException("master data source not found"));
        slaveFactory = createFactory(SLAVE, beanLocator, objectMapper, slaveSqlLogger);
    }

    public <R extends Repository> R createRepository(Class<R> repositoryType) {
        return masterFactory.createRepository(repositoryType);
    }

    public <R extends Repository> R createReadOnlyRepository(Class<R> repositoryType) {
        return slaveFactory.orElse(masterFactory).createRepository(repositoryType);
    }

    @Bean
    @Singleton
    public GeneralRepository generalRepository() {
        return createRepository(GeneralRepository.class);
    }

    @Bean
    @Singleton
    public ChangeLogRepository changeLogRepository() {
        return createRepository(ChangeLogRepository.class);
    }

    @Bean
    @Singleton
    public RoChangeLogRepository readOnlyChangeLogRepository() {
        return createReadOnlyRepository(RoChangeLogRepository.class);
    }

    @Bean
    @Singleton
    public ResourceTypeRepository resourceTypeRepository() {
        return createRepository(ResourceTypeRepository.class);
    }

    @Bean
    @Singleton
    public RoResourceTypeRepository readOnlyResourceTypeRepository() {
        return createReadOnlyRepository(RoResourceTypeRepository.class);
    }

    @Bean
    @Singleton
    public GroupRepository groupRepository() {
        return createRepository(GroupRepository.class);
    }

    @Bean
    @Singleton
    public RoGroupRepository readOnlyGroupRepository() {
        return createReadOnlyRepository(RoGroupRepository.class);
    }

    @Bean
    @Singleton
    public ResourceRepository resourceRepository() {
        return createRepository(ResourceRepository.class);
    }

    @Bean
    @Singleton
    public RoResourceRepository readOnlyResourceRepository() {
        return createReadOnlyRepository(RoResourceRepository.class);
    }

    @Bean
    @Singleton
    public UserRepository userRepository() {
        return createRepository(UserRepository.class);
    }

    @Bean
    @Singleton
    public RoUserRepository readOnlyUserRepository() {
        return createReadOnlyRepository(RoUserRepository.class);
    }

    @Bean
    @Singleton
    public TaskRepository taskRepository() {
        return createRepository(TaskRepository.class);
    }

    @Bean
    @Singleton
    public RoRoleRepository readOnlyRoleRepository() {
        return createReadOnlyRepository(RoleRepository.class);
    }

    @Bean
    @Singleton
    public RoleRepository roleRepository() {
        return createRepository(RoleRepository.class);
    }

    @Bean
    @Singleton
    public RoGrantRepository readOnlyGrantRepository() {
        return createReadOnlyRepository(RoGrantRepository.class);
    }

    @Bean
    @Singleton
    public GrantRepository grantRepository() {
        return createRepository(GrantRepository.class);
    }

    @Bean
    @Singleton
    public RoLocationRepository readOnlyLocationRepository() {
        return createReadOnlyRepository(RoLocationRepository.class);
    }

    @Bean
    @Singleton
    public LocationRepository locationRepository() {
        return createRepository(LocationRepository.class);
    }
}
