package ru.yandex.mail.cerberus.dao;

import one.util.streamex.StreamEx;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import ru.yandex.mail.cerberus.exception.GroupAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.GroupNotFoundException;
import ru.yandex.mail.cerberus.exception.LocationAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.LocationNotFoundException;
import ru.yandex.mail.cerberus.exception.ResourceAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.ResourceNotFoundException;
import ru.yandex.mail.cerberus.exception.ResourceTypeAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.ResourceTypeNotFoundException;
import ru.yandex.mail.cerberus.exception.RoleAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.RoleNotFoundException;
import ru.yandex.mail.cerberus.exception.UserAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.UserNotFoundException;

import javax.inject.Singleton;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

@Singleton
public class DefaultDataAccessExceptionMapper implements DataAccessExceptionMapper {
    private static final Map<String, Function<ServerErrorMessage, Throwable>> CONSTRAINT_VIOLATION_ERROR_MAPPING = Map.ofEntries(
        // cerberus.group_roles
        entry("group_roles_role_id_fkey", RoleNotFoundException::new),
        entry("group_roles_group_id_fkey", GroupNotFoundException::new),
        // cerberus.groups
        entry("groups_pkey", GroupAlreadyExistsException::new),
        // cerberus.locations
        entry("locations_pkey", LocationAlreadyExistsException::new),
        // cerberus.resource
        entry("resource_pkey", ResourceAlreadyExistsException::new),
        entry("resource_type_fkey", ResourceTypeNotFoundException::new),
        entry("resource_location_type_fkey", LocationNotFoundException::new),
        // cerberus.resource_type
        entry("resource_type_pkey", ResourceTypeAlreadyExistsException::new),
        // cerberus.roles
        entry("roles_pkey", RoleAlreadyExistsException::new),
        // cerberus.user_groups
        entry("user_groups_uid_fkey", UserNotFoundException::new),
        entry("user_groups_group_id_fkey", GroupNotFoundException::new),
        // cerberus.user_roles
        entry("user_roles_role_id_fkey", RoleNotFoundException::new),
        // cerberus.users
        entry("users_pkey", UserAlreadyExistsException::new),
        // cerberus.grants
        entry("grants_uid_fkey", UserNotFoundException::new),
        entry("grants_role_id_fkey", RoleNotFoundException::new),
        entry("grants_resource_type_fkey", ResourceTypeNotFoundException::new),
        entry("grants_resource_type_fkey1", ResourceNotFoundException::new),
        entry("grants_group_id_fkey", GroupNotFoundException::new)
    );

    private static <E extends Throwable> Map.Entry<String, Function<ServerErrorMessage, E>> entry(String constraintName,
                                                                                                  Function<String, E> constructor) {
        return Map.entry(constraintName, msg -> constructor.apply(msg.getDetail()));
    }

    private static Optional<Throwable> mapConstraintViolation(ServerErrorMessage message) {
        return Optional.ofNullable(CONSTRAINT_VIOLATION_ERROR_MAPPING.get(message.getConstraint()))
            .map(mapper -> mapper.apply(message));
    }

    @Override
    public Optional<Throwable> map(DataIntegrityViolationException e) {
        return StreamEx.of(e.getRootCause())
            .select(PSQLException.class)
            .map(PSQLException::getServerErrorMessage)
            .map(DefaultDataAccessExceptionMapper::mapConstraintViolation)
            .flatMap(StreamEx::of)
            .findFirst();
    }
}
