package ru.yandex.mail.cerberus.dao.grant;

import org.jdbi.v3.sqlobject.customizer.Bind;
import org.jdbi.v3.sqlobject.statement.GetGeneratedKeys;
import org.jdbi.v3.sqlobject.statement.SqlQuery;
import org.jdbi.v3.sqlobject.statement.SqlUpdate;
import ru.yandex.mail.cerberus.GrantId;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.asyncdb.CrudRepository;
import ru.yandex.mail.cerberus.asyncdb.annotations.ConfigureCrudRepository;

import java.util.Optional;
import java.util.Set;

@ConfigureCrudRepository(table = "cerberus.grants")
public interface GrantRepository extends RoGrantRepository, CrudRepository<GrantId, GrantEntity> {
    @GetGeneratedKeys
    @SqlUpdate("INSERT INTO <table> (uid, resource_type, resource_id, actions)\n"
             + "VALUES (:uid, :resourceType, :resourceId, :actions)\n"
             + "ON CONFLICT (uid, resource_type, resource_id)\n"
             + "  WHERE uid IS NOT NULL\n"
             + "  DO UPDATE SET actions = EXCLUDED.actions\n"
             + "RETURNING *")
    GrantEntity upsert(Uid uid, ResourceTypeName resourceType, Optional<ResourceId> resourceId, @Bind Set<String> actions);

    @GetGeneratedKeys
    @SqlUpdate("INSERT INTO <table> (role_id, resource_type, resource_id, actions)\n"
             + "VALUES (:roleId, :resourceType, :resourceId, :actions)\n"
             + "ON CONFLICT (role_id, resource_type, resource_id)\n"
             + "  WHERE role_id IS NOT NULL\n"
             + "  DO UPDATE SET actions = EXCLUDED.actions\n"
             + "RETURNING *")
    GrantEntity upsert(RoleId roleId, ResourceTypeName resourceType, Optional<ResourceId> resourceId, @Bind Set<String> actions);

    @GetGeneratedKeys
    @SqlUpdate("INSERT INTO <table> (group_id, group_type, resource_type, resource_id, actions)\n"
             + "VALUES (:groupId, :groupType, :resourceType, :resourceId, :actions)\n"
             + "ON CONFLICT (group_id, group_type, resource_type, resource_id)\n"
             + "  WHERE group_id IS NOT NULL AND group_type IS NOT NULL\n"
             + "  DO UPDATE SET actions = EXCLUDED.actions\n"
             + "RETURNING *")
    GrantEntity upsert(GroupId groupId, GroupType groupType, ResourceTypeName resourceType, Optional<ResourceId> resourceId,
                       @Bind Set<String> actions);

    default GrantEntity upsert(GroupKey groupKey, ResourceTypeName resourceType, Optional<ResourceId> resourceId,
                               Set<String> actions) {
        return upsert(groupKey.getId(), groupKey.getType(), resourceType, resourceId, actions);
    }

    @GetGeneratedKeys("id")
    @SqlQuery("DELETE FROM <table>\n"
            + "WHERE uid = :uid AND resource_type = :resourceType AND resource_id IS NULL\n"
            + "RETURNING id")
    Optional<GrantId> delete(Uid uid, ResourceTypeName resourceType);

    @GetGeneratedKeys("id")
    @SqlQuery("DELETE FROM <table>\n"
            + "WHERE role_id = :roleId AND resource_type = :resourceType AND resource_id IS NULL\n"
            + "RETURNING id")
    Optional<GrantId> delete(RoleId roleId, ResourceTypeName resourceType);

    @GetGeneratedKeys("id")
    @SqlUpdate("DELETE FROM <table>\n"
             + "WHERE group_id = :groupId AND group_type = :groupType AND resource_type = :resourceType AND resource_id IS NULL\n"
             + "RETURNING id")
    Optional<GrantId> delete(GroupId groupId, GroupType groupType, ResourceTypeName resourceType);

    default Optional<GrantId> delete(GroupKey groupKey, ResourceTypeName resourceType) {
        return delete(groupKey.getId(), groupKey.getType(), resourceType);
    }

    @GetGeneratedKeys("id")
    @SqlQuery("DELETE FROM <table>\n"
            + "WHERE uid = :uid AND resource_type = :type AND resource_id = :resourceId\n"
            + "RETURNING id")
    Optional<GrantId> delete(Uid uid, ResourceTypeName type, ResourceId resourceId);

    @GetGeneratedKeys("id")
    @SqlQuery("DELETE FROM <table>\n"
            + "WHERE role_id = :roleId AND resource_type = :type AND resource_id = :resourceId\n"
            + "RETURNING id")
    Optional<GrantId> delete(RoleId roleId, ResourceTypeName type, ResourceId resourceId);

    @GetGeneratedKeys("id")
    @SqlQuery("DELETE FROM <table>\n"
            + "WHERE group_id = :groupId AND group_type = :groupType AND resource_type = :type AND resource_id = :resourceId\n"
            + "RETURNING id")
    Optional<GrantId> delete(GroupId groupId, GroupType groupType, ResourceTypeName type, ResourceId resourceId);

    default Optional<GrantId> delete(GroupKey groupKey, ResourceTypeName resourceType, ResourceId resourceId) {
        return delete(groupKey.getId(), groupKey.getType(), resourceType, resourceId);
    }

    default Optional<GrantId> delete(Uid uid, ResourceTypeName type, Optional<ResourceId> resourceId) {
        return resourceId
            .flatMap(id -> delete(uid, type, id))
            .or(() -> delete(uid, type));
    }

    default Optional<GrantId> delete(RoleId roleId, ResourceTypeName type, Optional<ResourceId> resourceId) {
        return resourceId
            .flatMap(id -> delete(roleId, type, id))
            .or(() -> delete(roleId, type));
    }

    default Optional<GrantId> delete(GroupId groupId, GroupType groupType, ResourceTypeName type, Optional<ResourceId> resourceId) {
        return resourceId
            .flatMap(id -> delete(groupId, groupType, type, id))
            .or(() -> delete(groupId, groupType, type));
    }

    default Optional<GrantId> delete(GroupKey groupKey, ResourceTypeName type, Optional<ResourceId> resourceId) {
        return delete(groupKey.getId(), groupKey.getType(), type, resourceId);
    }
}
