package ru.yandex.mail.cerberus.dao.grant;

import lombok.val;
import one.util.streamex.StreamEx;
import org.jdbi.v3.sqlobject.config.RegisterConstructorMapper;
import org.jdbi.v3.sqlobject.customizer.BindList;
import org.jdbi.v3.sqlobject.statement.SqlQuery;
import ru.yandex.mail.cerberus.GrantId;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceKey;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.asyncdb.RoCrudRepository;
import ru.yandex.mail.cerberus.asyncdb.annotations.BindBeans;
import ru.yandex.mail.cerberus.asyncdb.annotations.ConfigureCrudRepository;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static java.util.Collections.emptyList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToMap;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

class ResourceActionsIndex {
    private final Map<ResourceIdEntity, List<String>> index;

    private List<String> findActionsList(ResourceTypeName resourceType) {
        val key = new ResourceIdEntity(resourceType);
        return index.getOrDefault(key, emptyList());
    }

    ResourceActionsIndex(List<ResourceActionsEntity> actions) {
        this.index = mapToMap(actions, ResourceIdEntity::new, ResourceActionsEntity::getActions);
    }

    ResourceActionsIndex(ResourceTypeName resourceType, List<ResourceIdActionsEntity> actions) {
        this.index = mapToMap(
            actions,
            entity -> new ResourceIdEntity(resourceType, Optional.of(entity.getResourceId())),
            ResourceIdActionsEntity::getActions
        );
    }

    Set<String> findActions(ResourceKey resourceKey) {
        val key = new ResourceIdEntity(resourceKey);
        val resourceActions = index.getOrDefault(key, emptyList());
        val resourceTypeActions = findActions(resourceKey.getType());

        return StreamEx.of(resourceActions)
            .append(resourceTypeActions)
            .toImmutableSet();
    }

    Set<String> findActions(ResourceTypeName resourceType) {
        return Set.copyOf(findActionsList(resourceType));
    }
}

@ConfigureCrudRepository(table = "cerberus.grants")
public interface RoGrantRepository extends RoCrudRepository<GrantId, GrantEntity> {
    @SqlQuery("WITH is_superuser AS (\n"
            + "    SELECT superuser\n"
            + "    FROM cerberus.users\n"
            + "    WHERE uid = :uid\n"
            + "    UNION ALL (SELECT FALSE)\n"
            + "    LIMIT 1\n"
            + ")\n"
            + "(\n"
            + "    SELECT sub.resource_type AS resource_type,\n"
            + "           sub.resource_id AS resource_id,\n"
            + "           resource_type.action_set AS actions\n"
            + "    FROM (VALUES <resourceIds>) AS sub (resource_id, resource_type)\n"
            + "    JOIN cerberus.resource_type resource_type ON resource_type.name = sub.resource_type\n"
            + "    WHERE (TABLE is_superuser)\n"
            + ")\n"
            + "UNION ALL\n"
            + "(\n"
            + "    SELECT resource_type,\n"
            + "           resource_id,\n"
            + "           array_agg(DISTINCT action) AS actions\n"
            + "    FROM (\n"
            + "        SELECT grants.resource_type AS resource_type,\n"
            + "               grants.resource_id AS resource_id,\n"
            + "               grants.actions AS actions\n"
            + "        FROM cerberus.users users\n"
            + "        LEFT OUTER JOIN cerberus.user_groups user_groups ON users.uid = user_groups.uid\n"
            + "        LEFT OUTER JOIN cerberus.user_roles user_roles ON users.uid = user_roles.uid\n"
            + "        LEFT OUTER JOIN cerberus.group_roles group_roles\n"
            + "            ON user_groups.group_id = group_roles.group_id AND user_groups.group_type = group_roles.group_type\n"
            + "        JOIN cerberus.grants grants\n"
            + "            ON ((users.uid = grants.uid) OR\n"
            + "                (user_groups.group_id = grants.group_id AND user_groups.group_type = grants.group_type) OR\n"
            + "                (user_roles.role_id = grants.role_id) OR\n"
            + "                (group_roles.role_id = grants.role_id))\n"
            + "            AND (((grants.resource_id, grants.resource_type) IN (<resourceIds>)) OR\n"
            + "                 (grants.resource_id IS NULL AND grants.resource_type IN (SELECT type FROM (VALUES <resourceIds>) AS res (id, type))))\n"
            + "        WHERE NOT (TABLE is_superuser) AND users.uid = :uid\n"
            + "        ORDER BY resource_type, resource_id\n"
            + "    ) sub, unnest(actions) AS action\n"
            + "    GROUP BY resource_type, resource_id\n"
            + ")")
    @RegisterConstructorMapper(ResourceActionsEntity.class)
    List<ResourceActionsEntity> findExistingActions(Uid uid, @BindBeans Set<ResourceIdEntity> resourceIds);

    @SqlQuery("WITH is_superuser AS (\n"
            + "    SELECT superuser\n"
            + "    FROM cerberus.users\n"
            + "    WHERE uid = :uid\n"
            + "    UNION ALL (SELECT FALSE)\n"
            + "    LIMIT 1\n"
            + ")\n"
            + "(\n"
            + "    SELECT resource.id AS resource_id,\n"
            + "           resource_type.action_set AS actions\n"
            + "    FROM cerberus.resource resource\n"
            + "    JOIN cerberus.resource_type resource_type ON resource_type.name = resource.type\n"
            + "    WHERE (TABLE is_superuser) AND resource.id IN (<resourceIds>) AND resource.type = :resourceType\n"
            + ")\n"
            + "UNION ALL\n"
            + "(\n"
            + "    SELECT resource_id,\n"
            + "           array_agg(DISTINCT action) AS actions\n"
            + "    FROM (\n"
            + "        SELECT grants.resource_id AS resource_id,\n"
            + "               grants.actions AS actions\n"
            + "        FROM cerberus.users users\n"
            + "        LEFT OUTER JOIN cerberus.user_groups user_groups ON users.uid = user_groups.uid\n"
            + "        LEFT OUTER JOIN cerberus.user_roles user_roles ON users.uid = user_roles.uid\n"
            + "        LEFT OUTER JOIN cerberus.group_roles group_roles\n"
            + "            ON user_groups.group_id = group_roles.group_id AND user_groups.group_type = group_roles.group_type\n"
            + "        JOIN cerberus.grants grants\n"
            + "            ON ((users.uid = grants.uid) OR\n"
            + "                (user_groups.group_id = grants.group_id AND user_groups.group_type = grants.group_type) OR\n"
            + "                (user_roles.role_id = grants.role_id) OR\n"
            + "                (group_roles.role_id = grants.role_id))\n"
            + "            AND ((grants.resource_type = :resourceType AND grants.resource_id IN (<resourceIds>)) OR\n"
            + "                 (grants.resource_id IS NULL AND grants.resource_type = :resourceType))\n"
            + "        WHERE NOT (TABLE is_superuser) AND users.uid = :uid\n"
            + "        ORDER BY resource_id\n"
            + "    ) sub, unnest(actions) AS action\n"
            + "    GROUP BY resource_id\n"
            + ")")
    @RegisterConstructorMapper(ResourceIdActionsEntity.class)
    List<ResourceIdActionsEntity> findExistingActions(Uid uid, ResourceTypeName resourceType, @BindList Set<ResourceId> resourceIds);

    default List<ResourceActionsInfo> findResourceActions(Uid uid, Set<ResourceKey> keys) {
        val ids = mapToSet(keys, ResourceIdEntity::new);
        val index = new ResourceActionsIndex(findExistingActions(uid, ids));

        return StreamEx.of(keys)
            .map(key -> {
                val actions = index.findActions(key);
                val resourceId = key.getId();
                return new ResourceActionsInfo(resourceId, key.getType(), actions);
            })
            .toImmutableList();
    }

    default List<ResourceActionsInfo> findResourceActions(Uid uid, ResourceTypeName resourceType, Set<ResourceId> ids) {
        val index = new ResourceActionsIndex(resourceType, findExistingActions(uid, resourceType, ids));

        return StreamEx.of(ids)
            .map(id -> {
                val key = new ResourceKey(id, resourceType);
                val actions = index.findActions(key);
                return new ResourceActionsInfo(id, resourceType, actions);
            })
            .toImmutableList();
    }

    default List<ResourceTypeActionsInfo> findResourceTypeActions(Uid uid, Set<ResourceTypeName> types) {
        val ids = mapToSet(types, ResourceIdEntity::new);
        val index = new ResourceActionsIndex(findExistingActions(uid, ids));

        return StreamEx.of(types)
            .map(type -> {
                val actions = index.findActions(type);
                return new ResourceTypeActionsInfo(type, actions);
            })
            .toImmutableList();
    }
}
