package ru.yandex.mail.cerberus.dao.task;

import lombok.Value;
import lombok.val;
import org.jdbi.v3.core.mapper.Nested;
import org.jdbi.v3.sqlobject.config.KeyColumn;
import org.jdbi.v3.sqlobject.config.RegisterConstructorMapper;
import org.jdbi.v3.sqlobject.config.RegisterConstructorMappers;
import org.jdbi.v3.sqlobject.config.ValueColumn;
import org.jdbi.v3.sqlobject.customizer.BindList;
import org.jdbi.v3.sqlobject.statement.GetGeneratedKeys;
import org.jdbi.v3.sqlobject.statement.SqlQuery;
import org.jdbi.v3.sqlobject.statement.SqlUpdate;
import ru.yandex.mail.cerberus.IdempotencyKey;
import ru.yandex.mail.cerberus.TaskType;
import ru.yandex.mail.cerberus.asyncdb.Repository;
import ru.yandex.mail.cerberus.asyncdb.annotations.JsonB;

import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

@RegisterConstructorMappers({
    @RegisterConstructorMapper(TaskRepository.GenericTaskInfo.class),
    @RegisterConstructorMapper(TaskRepository.RunningTaskInfo.class),
    @RegisterConstructorMapper(TaskRepository.RecentTaskInfo.class)
})
public interface TaskRepository extends Repository {
    @Value
    class GenericTaskInfo {
        IdempotencyKey idempotencyKey;
        TaskType type;
        OffsetDateTime schedule;
        OffsetDateTime created;
        Duration timeout;
        TaskStatus status;
        String requestId;
        Optional<Long> initiatorUid;
        Optional<String> context;
    }

    @Value
    class RunningTaskInfo {
        @Nested GenericTaskInfo info;
        String host;
        OffsetDateTime started;
    }

    @Value
    class RecentTaskInfo {
        IdempotencyKey idempotencyKey;
        TaskType type;
        OffsetDateTime created;
        OffsetDateTime started;
        OffsetDateTime finished;
        TaskStatus status;
        String requestId;
        Optional<Long> initiatorUid;
        Optional<String> error;
        Optional<String> context;
    }

    @GetGeneratedKeys
    @SqlUpdate("INSERT INTO cerberus.tasks (idempotency_key, type, schedule, timeout, status, request_id, initiator_uid, host, context)\n"
             + "VALUES (:idempotencyKey, :type, current_timestamp + :delay, :timeout, 'NEW', :requestId, :initiatorUid, :host, :context)")
    <T> GenericTaskInfo insertTask(IdempotencyKey idempotencyKey, TaskType type, Duration delay, Duration timeout, String requestId,
                                   Optional<Long> initiatorUid, Optional<String> host, @JsonB T context);

    default <T> GenericTaskInfo insertTask(IdempotencyKey idempotencyKey, TaskType type, Duration delay, Duration timeout,
                                           String requestId, Optional<Long> initiatorUid, Optional<String> host, Optional<T> context) {
        return insertTask(idempotencyKey, type, delay, timeout, requestId, initiatorUid, host, context.orElse(null));
    }

    @GetGeneratedKeys
    @SqlUpdate("INSERT INTO cerberus.recent_tasks\n"
             + "(idempotency_key, type, created, started, status, request_id, initiator_uid, error, host, context)\n"
             + "VALUES (:idempotencyKey, :type, :created, :started, :status, :requestId, :initiatorUid, :error, :host, :context)")
    RecentTaskInfo insertRecentTask(IdempotencyKey idempotencyKey, TaskType type, OffsetDateTime created, OffsetDateTime started,
                                    TaskStatus status, String requestId, Optional<Long> initiatorUid, Optional<String> error,
                                    String host, @JsonB String context);

    default RecentTaskInfo insertRecentTask(IdempotencyKey idempotencyKey, TaskType type, OffsetDateTime created, OffsetDateTime started,
                                            TaskStatus status, String requestId, Optional<Long> initiatorUid, Optional<String> error,
                                            String host, Optional<String> context) {
        return insertRecentTask(idempotencyKey, type, created, started, status, requestId, initiatorUid, error, host,
            context.orElse(null));
    }

    @SqlQuery("SELECT idempotency_key, type, schedule, created, timeout, status, request_id, initiator_uid, context\n"
            + "FROM cerberus.tasks\n"
            + "WHERE idempotency_key = :idempotencyKey")
    Optional<GenericTaskInfo> findTask(IdempotencyKey idempotencyKey);

    @SqlQuery("SELECT idempotency_key, type, schedule, created, timeout, status, request_id, initiator_uid, context\n"
            + "FROM cerberus.tasks\n"
            + "WHERE type = :type\n"
            + "LIMIT 1")
    Optional<GenericTaskInfo> findTask(TaskType type);

    @SqlQuery("UPDATE cerberus.tasks\n"
            + "SET started = current_timestamp,\n"
            + "    status = 'STARTED',\n"
            + "    host = :host\n"
            + "WHERE idempotency_key IN (\n"
            + "  SELECT idempotency_key FROM cerberus.tasks\n"
            + "  WHERE status = 'NEW' AND schedule <= current_timestamp\n"
            + "  LIMIT :count\n"
            + "  FOR UPDATE SKIP LOCKED\n"
            + ")\n"
            + "RETURNING idempotency_key, type, schedule, created, timeout, status, request_id, initiator_uid, context")
    List<GenericTaskInfo> acquireTasks(int count, String host);

    @SqlQuery("DELETE FROM cerberus.tasks\n"
            + "WHERE idempotency_key = :idempotencyKey AND status = 'STARTED'\n"
            + "RETURNING idempotency_key, type, schedule, created, started, timeout, status, request_id, initiator_uid, host, context")
    Optional<RunningTaskInfo> removeRunningTask(IdempotencyKey idempotencyKey);

    default boolean finishTask(IdempotencyKey idempotencyKey, TaskStatus status, Optional<String> error) {
        val taskOpt = removeRunningTask(idempotencyKey);
        taskOpt.ifPresent(task -> {
            val info = task.info;
            insertRecentTask(info.getIdempotencyKey(), info.getType(), info.getCreated(), task.getStarted(), status,
                info.getRequestId(), info.getInitiatorUid(), error, task.getHost(), info.getContext());
        });
        return taskOpt.isPresent();
    }

    @SqlQuery("SELECT idempotency_key, type, schedule, created, started, timeout, status, request_id, initiator_uid, host, context\n"
            + "FROM cerberus.tasks\n"
            + "WHERE status = 'STARTED' AND (current_timestamp - started) > timeout\n"
            + "FOR UPDATE SKIP LOCKED\n"
            + "LIMIT 1")
    Optional<RunningTaskInfo> acquireExpiredTask();

    @SqlQuery("SELECT idempotency_key, type, created, started, finished, status, request_id, initiator_uid, error, context\n"
            + "FROM cerberus.recent_tasks\n"
            + "WHERE type = :type")
    List<RecentTaskInfo> findExecutedTasks(TaskType type);

    @KeyColumn("type")
    @ValueColumn("status")
    @SqlQuery("SELECT DISTINCT ON (type) type, status "
            + "FROM cerberus.recent_tasks "
            + "WHERE type IN (<types>) "
            + "ORDER BY type, finished DESC")
    Map<TaskType, TaskStatus> findLastTasksStatus(@BindList Set<TaskType> types);
}
