package ru.yandex.mail.cerberus.dao.user;

import lombok.val;
import org.jdbi.v3.sqlobject.config.KeyColumn;
import org.jdbi.v3.sqlobject.config.RegisterConstructorMapper;
import org.jdbi.v3.sqlobject.config.ValueColumn;
import org.jdbi.v3.sqlobject.customizer.BindList;
import org.jdbi.v3.sqlobject.statement.SqlQuery;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.asyncdb.Condition;
import ru.yandex.mail.cerberus.asyncdb.RoCrudRepository;
import ru.yandex.mail.cerberus.asyncdb.annotations.ConfigureCrudRepository;
import ru.yandex.mail.cerberus.asyncdb.util.OneToMany;

import java.util.Optional;
import java.util.Set;

import static java.util.Collections.singletonList;

@ConfigureCrudRepository(table = "cerberus.users")
public interface RoUserRepository extends RoCrudRepository<Uid, UserEntity> {
    default Page<Uid, UserEntity> findPage(Pageable<Uid> pageable) {
        return findPage(pageable, UserEntity::getUid);
    }

    default Page<Uid, UserEntity> findPageByType(Pageable<Uid> pageable, UserType type) {
        val condition = Condition.of("type = :type").bind("type", type);
        return findPage(pageable, UserEntity::getUid, condition);
    }

    @SqlQuery("SELECT superuser FROM <table>\n"
            + "WHERE <id> = :uid")
    Optional<Boolean> isSuperuser(Uid uid);

    @KeyColumn("uid")
    @ValueColumn("group_id")
    @SqlQuery("SELECT users.uid, user_groups.group_id\n"
            + "FROM <table> users\n"
            + "LEFT OUTER JOIN cerberus.user_groups user_groups ON (user_groups.uid = users.uid AND user_groups.group_type = :groupType)\n"
            + "WHERE users.uid IN (<uids>)")
    OneToMany<Uid, GroupId> findUserGroupsByType(@BindList Iterable<Uid> uids, GroupType groupType);

    @KeyColumn("uid")
    @SqlQuery("SELECT users.uid, user_groups.group_id AS id, user_groups.group_type AS type\n"
            + "FROM <table> users\n"
            + "LEFT OUTER JOIN cerberus.user_groups user_groups ON (user_groups.uid = users.uid)\n"
            + "WHERE users.uid IN (<uids>)")
    @RegisterConstructorMapper(GroupKey.class)
    OneToMany<Uid, GroupKey> findUsersGroupKeys(@BindList Iterable<Uid> uids);

    @KeyColumn("uid")
    @ValueColumn("role_id")
    @SqlQuery("SELECT users.uid, user_roles.role_id\n"
            + "FROM <table> users\n"
            + "LEFT OUTER JOIN cerberus.user_roles user_roles ON (user_roles.uid = users.uid)\n"
            + "WHERE users.uid IN (<uids>)")
    OneToMany<Uid, RoleId> findUserRoles(@BindList Iterable<Uid> uids);

    default Optional<Set<RoleId>> findUserRoles(Uid uid) {
        return Optional.ofNullable(findUserRoles(singletonList(uid)).getSetMapping().get(uid));
    }
}
