package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceKey;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.AllowedActions;
import ru.yandex.mail.cerberus.client.dto.AllowedTypeActions;
import ru.yandex.mail.cerberus.client.dto.Grant;
import ru.yandex.mail.cerberus.client.ops.GrantOperations;
import ru.yandex.mail.cerberus.core.grant.GrantManager;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;
import java.util.Set;

import static java.util.Objects.requireNonNullElseGet;

@TvmSecured
@Validated
@Controller("/grant")
@AllArgsConstructor(onConstructor_= @Inject)
public class GrantController implements GrantOperations {
    private final GrantManager grantManager;

    private ReadTarget resolve(@Nullable ReadTarget readTarget) {
        return requireNonNullElseGet(readTarget, grantManager::defaultReadTarget);
    }

    @Override
    public Mono<Void> setUserGrant(Uid uid, Grant grant) {
        return Mono.fromFuture(grantManager.setGrant(uid, grant));
    }

    @Override
    public Mono<Void> setGroupGrant(GroupId id, GroupType type, Grant grant) {
        return Mono.fromFuture(grantManager.setGrant(new GroupKey(id, type), grant));
    }

    @Override
    public Mono<Void> setRoleGrant(RoleId id, Grant grant) {
        return Mono.fromFuture(grantManager.setGrant(id, grant));
    }

    @Override
    public Mono<AllowedActions> actions(Uid uid, Set<ResourceKey> resources, @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(grantManager.actions(uid, resources, resolve(readTarget)));
    }

    @Override
    public Mono<AllowedActions> actions(Uid uid, ResourceTypeName resourceType, Set<ResourceId> resources,
                                        @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(grantManager.actions(uid, resourceType, resources, resolve(readTarget)));
    }

    @Override
    public Mono<AllowedTypeActions> typesActions(Uid uid, Set<ResourceTypeName> names, @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(grantManager.typesActions(uid, names, resolve(readTarget)));
    }
}
