package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.GroupData;
import ru.yandex.mail.cerberus.client.dto.Group;
import ru.yandex.mail.cerberus.client.ops.GroupOperations;
import ru.yandex.mail.cerberus.core.group.GroupManager;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;

import java.util.Optional;

import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNullElseGet;

@TvmSecured
@Validated
@Controller("/group")
@AllArgsConstructor(onConstructor_= @Inject)
public class GroupController implements GroupOperations {
    private final GroupManager groupManager;

    private ReadTarget resolve(@Nullable ReadTarget readTarget) {
        return requireNonNullElseGet(readTarget, groupManager::defaultReadTarget);
    }

    @Override
    public Mono<Group<RawJsonString>> addGroup(Group<RawJsonString> group) {
        return Mono.fromFuture(groupManager.insert(group));
    }

    @Override
    public Mono<Group<RawJsonString>> createGroup(GroupData<RawJsonString> data) {
        return Mono.fromFuture(groupManager.create(data));
    }

    @Override
    public Mono<Void> updateGroup(Group<RawJsonString> group) {
        return Mono.fromFuture(groupManager.update(singletonList(group)));
    }

    @Override
    public Mono<Void> deleteGroup(@Nullable Boolean strict, GroupId id, GroupType type) {
        return Mono.fromFuture(groupManager.delete(new GroupKey(id, type), DeletionMode.resolve(strict)));
    }

    @Override
    public Mono<Page<GroupKey, Group<RawJsonString>>> allGroups(int pageSize, @Nullable GroupKey pageId,
                                                                @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(groupManager.groups(new Pageable<>(Optional.ofNullable(pageId), pageSize),
            RawJsonString.class, resolve(readTarget)));
    }

    @Override
    public Mono<Page<GroupId, Group<RawJsonString>>> findGroupsByType(GroupType type, int pageSize,
                                                                      @Nullable GroupId pageId,
                                                                      @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(groupManager.groups(type, new Pageable<>(Optional.ofNullable(pageId), pageSize),
            RawJsonString.class, resolve(readTarget)));
    }

    @Override
    public Mono<Void> addUserToGroup(GroupId id, GroupType type, Uid uid) {
        return Mono.fromFuture(groupManager.addUser(new GroupKey(id, type), uid));
    }

    @Override
    public Mono<Void> removeUserFromGroup(GroupId id, GroupType type, Uid uid) {
        return Mono.fromFuture(groupManager.removeUser(new GroupKey(id, type), uid));
    }
}
