package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.LocationKey;
import ru.yandex.mail.cerberus.LocationType;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.client.dto.Location;
import ru.yandex.mail.cerberus.client.ops.LocationOperations;
import ru.yandex.mail.cerberus.core.location.LocationManager;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;

import java.util.List;
import java.util.Optional;

import static java.util.Objects.requireNonNullElseGet;

@TvmSecured
@Validated
@Controller("/location")
@AllArgsConstructor(onConstructor_= @Inject)
public class LocationController implements LocationOperations {
    private final LocationManager locationManager;

    private ReadTarget resolve(@Nullable ReadTarget readTarget) {
        return requireNonNullElseGet(readTarget, locationManager::defaultReadTarget);
    }

    @Override
    public Mono<Location<RawJsonString>> createLocation(Location<RawJsonString> location) {
        return Mono.fromFuture(locationManager.insert(location));
    }

    @Override
    public Mono<Void> updateLocation(Location<RawJsonString> location) {
        return Mono.fromFuture(locationManager.update(List.of(location)));
    }

    @Override
    public Mono<Void> deleteLocation(@Nullable Boolean strict, LocationId id, LocationType type) {
        return Mono.fromFuture(locationManager.delete(new LocationKey(id, type), DeletionMode.resolve(strict)));
    }

    @Override
    public Mono<Page<LocationKey, Location<RawJsonString>>> allLocations(int pageSize, @Nullable LocationKey pageId,
                                                                         @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(locationManager.locations(new Pageable<>(Optional.ofNullable(pageId), pageSize),
            RawJsonString.class, resolve(readTarget)));
    }

    @Override
    public Mono<Page<LocationId, Location<RawJsonString>>> findLocationsByType(LocationType type, int pageSize,
                                                                               @Nullable LocationId pageId,
                                                                               @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(locationManager.locationsByType(type, new Pageable<>(Optional.ofNullable(pageId), pageSize),
            RawJsonString.class, resolve(readTarget)));
    }
}
