package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.http.annotation.Delete;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import one.util.streamex.StreamEx;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.ResourceKey;
import ru.yandex.mail.cerberus.ResourceTypeName;
import ru.yandex.mail.cerberus.client.dto.Resource;
import ru.yandex.mail.cerberus.client.dto.ResourceData;
import ru.yandex.mail.cerberus.client.ops.ResourceOperations;
import ru.yandex.mail.cerberus.core.resource.ResourceManager;
import ru.yandex.mail.cerberus.client.dto.ResourceType;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNullElseGet;
import static java.util.function.Function.identity;

@Validated
@TvmSecured
@Controller("/resource")
@AllArgsConstructor(onConstructor_= @Inject)
public class ResourceController implements ResourceOperations {
    private final ResourceManager resourceManager;

    private ReadTarget resolve(@Nullable ReadTarget readTarget) {
        return requireNonNullElseGet(readTarget, resourceManager::defaultReadTarget);
    }

    @Override
    public Mono<ResourceType> createResourceType(ResourceType data) {
        return Mono.fromFuture(resourceManager.createType(data));
    }

    @Override
    public Mono<ResourceType> getOrCreateResourceType(ResourceType data) {
        return Mono.fromFuture(resourceManager.getByNameOrCreateType(data));
    }

    @Override
    public Mono<Optional<ResourceType>> findType(ResourceTypeName name, @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(resourceManager.getTypes(singleton(name), resolve(readTarget)))
            .map(types -> StreamEx.of(types).findFirst());
    }

    @Override
    public Mono<Map<ResourceTypeName, ResourceType>> findTypes(Set<ResourceTypeName> names, @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(resourceManager.getTypes(names, resolve(readTarget)))
            .map(types -> StreamEx.of(types).toMap(ResourceType::getName, identity()));
    }

    @Override
    public Mono<Page<ResourceId, Resource<RawJsonString>>> findResourcesByType(ResourceTypeName type,
                                                                               int pageSize,
                                                                               @Nullable ResourceId pageId,
                                                                               @Nullable ReadTarget readTarget) {
        return Mono.fromFuture(resourceManager.resources(type, new Pageable<>(Optional.ofNullable(pageId), pageSize),
            RawJsonString.class, resolve(readTarget)));
    }

    @Override
    public Mono<Resource<RawJsonString>> createResource(ResourceData<RawJsonString> data) {
        return Mono.fromFuture(resourceManager.createResource(data));
    }

    @Override
    public Mono<List<Resource<RawJsonString>>> createResources(List<ResourceData<RawJsonString>> data) {
        return Mono.fromFuture(resourceManager.createResources(data, RawJsonString.class));
    }

    @Override
    public Mono<Resource<RawJsonString>> addResource(@Nullable Boolean skipExisting,
                                                     Resource<RawJsonString> resource) {
        return Mono.fromFuture(resourceManager.insertResources(CollisionStrategy.resolve(skipExisting), singletonList(resource)))
            .map(list -> list.get(0));
    }

    @Override
    public Mono<List<Resource<RawJsonString>>> addResources(@Nullable Boolean skipExisting,
                                                            List<Resource<RawJsonString>> resources) {
        return Mono.fromFuture(resourceManager.insertResources(CollisionStrategy.resolve(skipExisting), resources));
    }

    @Override
    public Mono<Void> updateResource(Resource<RawJsonString> resource) {
        return Mono.fromFuture(resourceManager.updateResource(resource));
    }

    @Override
    public Mono<Void> updateResources(List<Resource<RawJsonString>> resources) {
        return Mono.fromFuture(resourceManager.updateResources(resources));
    }

    @Delete
    public Mono<Void> deleteResource(@Nullable Boolean strict, ResourceTypeName type, ResourceId id) {
        return Mono.fromFuture(resourceManager.deleteResource(new ResourceKey(id, type), DeletionMode.resolve(strict)));
    }

    @Override
    public Mono<Void> deleteResources(@Nullable Boolean strict, ResourceTypeName type, Set<ResourceId> id) {
        return Mono.fromFuture(resourceManager.deleteResources(type, id, DeletionMode.resolve(strict)));
    }
}
