package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.GroupType;
import ru.yandex.mail.cerberus.RoleId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.Role;
import ru.yandex.mail.cerberus.client.dto.RoleData;
import ru.yandex.mail.cerberus.client.ops.RoleOperations;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.cerberus.core.role.RoleManager;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;

@TvmSecured
@Validated
@Controller("/role")
@AllArgsConstructor(onConstructor_= @Inject)
public class RoleController implements RoleOperations {
    private final RoleManager roleManager;

    @Override
    public Mono<Role> createRole(RoleData data) {
        return Mono.fromFuture(roleManager.create(data));
    }

    @Override
    public Mono<Void> updateRole(Role role) {
        return Mono.fromFuture(roleManager.update(role));
    }

    @Override
    public Mono<Void> deleteRole(@Nullable Boolean strict, RoleId id) {
        return Mono.fromFuture(roleManager.delete(id, DeletionMode.resolve(strict)));
    }

    @Override
    public Mono<Void> assignRoleToUser(RoleId id, Uid uid) {
        return Mono.fromFuture(roleManager.attachRoleToUser(uid, id));
    }

    @Override
    public Mono<Void> removeRoleFromUser(RoleId id, Uid uid) {
        return Mono.fromFuture(roleManager.detachRoleFromUser(uid, id));
    }

    @Override
    public Mono<Void> assignRoleToGroup(RoleId id, GroupId groupId, GroupType groupType) {
        return Mono.fromFuture(roleManager.attachRoleToGroup(new GroupKey(groupId, groupType), id));
    }

    @Override
    public Mono<Void> removeRoleFromGroup(RoleId id, GroupId groupId, GroupType groupType) {
        return Mono.fromFuture(roleManager.detachRoleFromGroup(new GroupKey(groupId, groupType), id));
    }
}
