package ru.yandex.mail.cerberus.controller;

import io.micronaut.http.annotation.Controller;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import lombok.val;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.client.dto.UserWithGroups;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.DeletionMode;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.cerberus.client.ops.UserOperations;
import ru.yandex.mail.cerberus.core.user.UserManager;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.annotation.Nullable;
import javax.inject.Inject;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import static java.util.Objects.requireNonNullElseGet;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@Validated
@TvmSecured
@Controller("/user")
@AllArgsConstructor(onConstructor_= @Inject)
public class UserController implements UserOperations {
    private final UserManager userManager;

    private ReadTarget resolve(@Nullable ReadTarget readTarget) {
        return requireNonNullElseGet(readTarget, userManager::defaultReadTarget);
    }

    @Override
    public Mono<Uid> addUser(User<RawJsonString> user) {
        return Mono.fromFuture(userManager.insert(user).thenApply(User::getUid));
    }

    @Override
    public Mono<Set<Uid>> addUsers(@Nullable Boolean skipExisting, List<User<RawJsonString>> users) {
        return Mono.fromFuture(userManager.insert(CollisionStrategy.resolve(skipExisting), users))
            .map(resultList -> mapToSet(resultList, User::getUid));
    }

    @Override
    public Mono<Void> updateUser(User<RawJsonString> user) {
        return Mono.fromFuture(userManager.update(user));
    }

    @Override
    public Mono<Void> deleteUser(@Nullable Boolean strict, Uid uid) {
        return Mono.fromFuture(userManager.delete(uid, DeletionMode.resolve(strict)));
    }

    @Override
    public Mono<Page<Uid, User<RawJsonString>>> users(@Nullable UserType type, int pageSize, @Nullable Uid pageId,
                                                      @Nullable ReadTarget readTarget) {
        val pageable = new Pageable<>(Optional.ofNullable(pageId), pageSize);
        if (type != null) {
            return Mono.fromFuture(userManager.users(type, pageable, RawJsonString.class, resolve(readTarget)));
        } else {
            return Mono.fromFuture(userManager.users(pageable, RawJsonString.class, resolve(readTarget)));
        }
    }

    @Override
    public Mono<Page<Uid, UserWithGroups<RawJsonString>>> usersWithGroups(@Nullable UserType type,
                                                                          int pageSize,
                                                                          @Nullable Uid pageId,
                                                                          @Nullable ReadTarget readTarget) {
        val pageable = new Pageable<>(Optional.ofNullable(pageId), pageSize);
        if (type != null) {
            return Mono.fromFuture(userManager.usersWithGroups(type, pageable, RawJsonString.class, resolve(readTarget)));
        } else {
            return Mono.fromFuture(userManager.usersWithGroups(pageable, RawJsonString.class, resolve(readTarget)));
        }
    }
}
