package ru.yandex.mail.cerberus.error;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.micronaut.core.bind.exceptions.UnsatisfiedArgumentException;
import io.micronaut.core.convert.exceptions.ConversionErrorException;
import io.micronaut.http.HttpRequest;
import io.micronaut.http.HttpResponse;
import io.micronaut.http.HttpStatus;
import io.micronaut.http.MutableHttpResponse;
import io.micronaut.http.annotation.Filter;
import io.micronaut.http.exceptions.ContentLengthExceededException;
import io.micronaut.http.exceptions.HttpStatusException;
import io.micronaut.http.filter.HttpServerFilter;
import io.micronaut.http.filter.ServerFilterChain;
import io.micronaut.security.authentication.AuthenticationException;
import io.micronaut.web.router.exceptions.DuplicateRouteException;
import io.micronaut.web.router.exceptions.UnsatisfiedRouteException;
import lombok.AllArgsConstructor;
import lombok.val;
import one.util.streamex.StreamEx;
import org.reactivestreams.Publisher;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.TransientDataAccessException;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.ErrorCode;
import ru.yandex.mail.cerberus.client.dto.ErrorResult;
import ru.yandex.mail.cerberus.dao.DataAccessExceptionMapper;
import ru.yandex.mail.cerberus.exception.CerberusException;
import ru.yandex.mail.cerberus.exception.EntityAlreadyExistsException;
import ru.yandex.mail.cerberus.exception.EntityNotFoundException;
import ru.yandex.mail.cerberus.exception.InputException;
import ru.yandex.mail.cerberus.exception.NoDatabaseConnectionException;

import javax.inject.Inject;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ElementKind;
import java.util.concurrent.CompletionException;

@Filter("/**")
@AllArgsConstructor(onConstructor_=@Inject)
public class ErrorMappingFilter implements HttpServerFilter {
    private final DataAccessExceptionMapper dataAccessExceptionMapper;

    private static ErrorResult result(CerberusException e) {
        return new ErrorResult(e.getMessage(), e.getErrorCode());
    }

    private static ErrorResult requestErrorResult(Throwable e) {
        return new ErrorResult(e.getMessage(), ErrorCode.REQUEST_ERROR);
    }

    private static HttpResponse<ErrorResult> handle(CerberusException e) {
        if (e instanceof InputException || e instanceof EntityNotFoundException) {
            return HttpResponse.badRequest(result(e));
        } else if (e instanceof EntityAlreadyExistsException) {
            return HttpResponse.<ErrorResult>status(HttpStatus.CONFLICT).body(result(e));
        } else if (e instanceof NoDatabaseConnectionException) {
            return HttpResponse.<ErrorResult>status(HttpStatus.BANDWIDTH_LIMIT_EXCEEDED).body(result(e));
        } else {
            return HttpResponse.serverError(result(e));
        }
    }

    private static HttpResponse<ErrorResult> handleGeneric(DataAccessException e) {
        val error = new ErrorResult(e.getMessage(), ErrorCode.DATABASE_ERROR);
        if (e instanceof TransientDataAccessException) {
            return HttpResponse.<ErrorResult>status(HttpStatus.SERVICE_UNAVAILABLE).body(error);
        } else {
            return HttpResponse.serverError(error);
        }
    }

    private HttpResponse<ErrorResult> handle(DataAccessException e) {
        return dataAccessExceptionMapper.map(e)
            .map(this::handle)
            .orElseGet(() -> handleGeneric(e));
    }

    private static String buildMessage(ConstraintViolation violation) {
        val propertyPath = violation.getPropertyPath();

        val message = StreamEx.of(propertyPath.spliterator())
            .filter(node -> node.getKind() == ElementKind.METHOD || node.getKind() == ElementKind.CONSTRUCTOR)
            .joining(".");

        return message + ": " + violation.getMessage();
    }

    private static HttpResponse<ErrorResult> handle(ConstraintViolationException e) {
        val constraintViolations = e.getConstraintViolations();
        if (constraintViolations == null || constraintViolations.isEmpty()) {
            val message = (e.getMessage() == null) ? HttpStatus.BAD_REQUEST.getReason() : e.getMessage();
            return HttpResponse.badRequest(new ErrorResult(message, ErrorCode.REQUEST_ERROR));
        } else {
            val message = StreamEx.of(constraintViolations)
                .map(ErrorMappingFilter::buildMessage)
                .joining("\n");
            return HttpResponse.badRequest(new ErrorResult(message, ErrorCode.REQUEST_ERROR));
        }
    }

    private static HttpResponse<ErrorResult> handle(HttpStatusException e) {
        return HttpResponse.<ErrorResult>status(e.getStatus()).body(requestErrorResult(e));
    }

    private static HttpResponse<ErrorResult> handle(JsonProcessingException e) {
        return HttpResponse.badRequest(new ErrorResult("Invalid JSON: " + e.getMessage(), ErrorCode.REQUEST_ERROR));
    }

    private static HttpResponse<ErrorResult> handleCommon(Throwable e) {
        if (e instanceof AuthenticationException) {
            return HttpResponse.<ErrorResult>unauthorized().body(requestErrorResult(e));
        } else if (e instanceof ConstraintViolationException) {
            return handle((ConstraintViolationException) e);
        } else if (e instanceof ContentLengthExceededException) {
            return HttpResponse.<ErrorResult>status(HttpStatus.REQUEST_ENTITY_TOO_LARGE).body(requestErrorResult(e));
        } else if (e instanceof HttpStatusException) {
            return handle((HttpStatusException) e);
        } else if (e instanceof JsonProcessingException) {
            return handle((JsonProcessingException) e);
        } else if (e instanceof ConversionErrorException
            || e instanceof DuplicateRouteException
            || e instanceof UnsatisfiedArgumentException
            || e instanceof UnsatisfiedRouteException) {
            return HttpResponse.badRequest(requestErrorResult(e));
        } else {
            return HttpResponse.serverError(new ErrorResult(e.getMessage(), ErrorCode.INTERNAL_ERROR));
        }
    }

    private HttpResponse<ErrorResult> handle(Throwable e) {
        if (e instanceof CompletionException) {
            return handle(e.getCause());
        } else if (e instanceof CerberusException) {
            return handle((CerberusException) e);
        } else if (e instanceof DataAccessException) {
            return handle((DataAccessException) e);
        } else {
            return handleCommon(e);
        }
    }

    @Override
    public Publisher<MutableHttpResponse<?>> doFilter(HttpRequest<?> request, ServerFilterChain chain) {
        return Mono.from(chain.proceed(request))
            .onErrorMap(e -> new ServerErrorException(e, handle(e)));
    }

    @Override
    public int getOrder() {
        return LOWEST_PRECEDENCE;
    }
}
