package ru.yandex.mail.cerberus.worker.controller;

import io.micronaut.http.annotation.Body;
import io.micronaut.http.annotation.Controller;
import io.micronaut.http.annotation.Post;
import io.micronaut.http.annotation.QueryValue;
import io.micronaut.validation.Validated;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import ru.yandex.mail.cerberus.IdempotencyKey;
import ru.yandex.mail.cerberus.TaskType;
import ru.yandex.mail.cerberus.worker.api.TaskProcessor;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.cerberus.worker.api.SubmitResult;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;
import ru.yandex.mail.cerberus.worker.TaskRegistry;
import ru.yandex.mail.cerberus.worker.api.Task;
import ru.yandex.mail.cerberus.worker.api.TaskExecutor;
import ru.yandex.mail.cerberus.worker.exception.UnknownTaskException;

import javax.inject.Inject;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static ru.yandex.mail.micronaut.common.Async.asyncSafe;

@Slf4j
@Validated
@TvmSecured
@Controller("/task")
@AllArgsConstructor(onConstructor_= @Inject)
public class TaskController {
    public static final String EMPTY_CONTEXT = "{}";

    private final TaskExecutor taskExecutor;
    private final TaskRegistry taskRegistry;
    private final JsonMapper jsonMapper;

    private <T> T parseContext(String context, Class<T> contextType) {
        return jsonMapper.fromJson(context, contextType);
    }

    @Post("/execute")
    public CompletableFuture<SubmitResult> execute(@QueryValue TaskType type, @QueryValue Optional<IdempotencyKey> key,
                                                   @Body String context) {
        return asyncSafe(() -> {
            log.info("Incoming task execution request: type={}, context={}", type, context);

            final TaskProcessor<?> processor = taskRegistry.findTaskProcessor(type)
                .orElseThrow(() -> new UnknownTaskException(type));

            val contextType = processor.contextType();
            val contextInstance = (contextType.equals(Void.class) || context.equals(EMPTY_CONTEXT))
                ? Optional.empty()
                : Optional.of(parseContext(context, contextType));

            val idempotencyKey = key.orElseGet(IdempotencyKey::random);
            val task = new Task<>(type, idempotencyKey, contextInstance);

            return taskExecutor.submit(task, Optional.empty());
        });
    }
}
