package ru.yandex.mail.cerberus.worker.executer;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import ru.yandex.mail.micronaut.common.context.ContextManager;
import ru.yandex.mail.cerberus.worker.api.TaskConfiguration;
import ru.yandex.mail.cerberus.dao.tx.TxManager;
import ru.yandex.mail.micronaut.common.qualifier.Master;
import ru.yandex.mail.cerberus.worker.api.SubmitResult;
import ru.yandex.mail.cerberus.worker.api.Task;
import ru.yandex.mail.cerberus.worker.api.TaskExecutor;
import ru.yandex.mail.cerberus.dao.task.TaskRepository;
import ru.yandex.mail.cerberus.worker.TaskRegistry;
import ru.yandex.mail.cerberus.worker.exception.UnknownTaskException;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static ru.yandex.mail.micronaut.common.Async.asyncSafe;

@Slf4j
@Singleton
public class DefaultTaskExecutor implements TaskExecutor {
    private final TaskRegistry taskRegistry;
    private final TaskRepository taskRepository;
    private final TxManager txManager;

    @Inject
    public DefaultTaskExecutor(TaskRegistry taskRegistry, TaskRepository taskRepository, @Master TxManager txManager) {
        this.taskRegistry = taskRegistry;
        this.taskRepository = taskRepository;
        this.txManager = txManager;
    }

    private SubmitResult submitTask(Task<?> task, TaskConfiguration config, Optional<Long> initiatorUid, Duration delay) {
        val key = task.getIdempotencyKey();
        val type = task.getType();
        val requestId = ContextManager.currentRequestId();

        final var taskInfo = taskRepository.findTask(key).orElseGet(() -> {
            val newTask = taskRepository.insertTask(key, type, delay, config.getTimeout(), requestId, initiatorUid,
                Optional.empty(), task.getContext());
            log.info("{} task {} successfully submitted", type, key);
            return newTask;
        });

        return new SubmitResult(taskInfo);
    }

    private CompletableFuture<SubmitResult> submitRepeatable(Task<?> task, TaskConfiguration config, Optional<Long> initiatorUid,
                                                             Duration delay) {
        return txManager.executeAsync(() -> {
            val existing = taskRepository.findTask(task.getType());
            return existing.map(info ->{
                log.info("Found an existing task with status={}", info.getStatus());
                return new SubmitResult(info);
            }).orElseGet(() -> {
                val totalDelay = config.getRepetitionRate()
                    .orElse(Duration.ZERO)
                    .plus(delay);
                return submitTask(task, config, initiatorUid, totalDelay);
            });
        });
    }

    private CompletableFuture<SubmitResult> submitOneOff(Task<?> task, TaskConfiguration config, Optional<Long> initiatorUid,
                                                         Duration delay) {
        return txManager.executeAsync(() -> submitTask(task, config, initiatorUid, delay));
    }

    @Override
    public <Context> CompletableFuture<SubmitResult> submit(Task<Context> task, Optional<Long> initiatorUid, Duration delay) {
        return asyncSafe(() -> {
            val configuration = taskRegistry.findTaskConfiguration(task.getType())
                .orElseThrow(() -> new UnknownTaskException(task));

            if (configuration.isRepeatable()) {
                return submitRepeatable(task, configuration, initiatorUid, delay);
            } else {
                return submitOneOff(task, configuration, initiatorUid, delay);
            }
        });
    }
}
