package ru.yandex.mail.cerberus.worker.executer;

import io.micronaut.context.event.ApplicationEventListener;
import io.micronaut.context.event.StartupEvent;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.EntryStream;
import ru.yandex.mail.cerberus.TaskType;
import ru.yandex.mail.cerberus.dao.task.TaskRepository;
import ru.yandex.mail.cerberus.dao.task.TaskStatus;
import ru.yandex.mail.cerberus.dao.tx.TxManager;
import ru.yandex.mail.cerberus.worker.TaskRegistry;
import ru.yandex.mail.cerberus.worker.WorkerConfiguration;
import ru.yandex.mail.cerberus.worker.api.TaskConfiguration;
import ru.yandex.mail.micronaut.common.Async;
import ru.yandex.mail.micronaut.common.qualifier.Master;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.ea.async.Async.await;
import static ru.yandex.mail.cerberus.worker.executer.Worker.WORKER_EXECUTOR_NAME;

@Slf4j
@Singleton
public class Monitor implements ApplicationEventListener<StartupEvent> {
    private final ScheduledExecutorService workerExecutor;
    private final TaskRepository taskRepository;
    private final TxManager txManager;
    private final TaskRegistry taskRegistry;
    private final WorkerConfiguration workerConfiguration;

    @Inject
    public Monitor(@Named(WORKER_EXECUTOR_NAME) ExecutorService workerExecutor, TaskRepository taskRepository,
                   @Master TxManager txManager, TaskRegistry taskRegistry, WorkerConfiguration workerConfiguration) {
        this.workerExecutor = (ScheduledExecutorService) workerExecutor;
        this.taskRepository = taskRepository;
        this.txManager = txManager;
        this.taskRegistry = taskRegistry;
        this.workerConfiguration = workerConfiguration;
    }

    private CompletableFuture<Map<TaskType, TaskStatus>> findTasksStatus() {
        val types = taskRegistry.configurations()
            .map(TaskConfiguration::getTaskType)
            .toImmutableSet();

        return txManager.executeAsync(() -> taskRepository.findLastTasksStatus(types));
    }

    private CompletableFuture<Void> updateStats() {
        val statusByTask = await(findTasksStatus());

        val stateString = EntryStream.of(statusByTask)
            .join(" = ")
            .joining(", ");
        log.info("Tasks state: {}", stateString);

        EntryStream.of(statusByTask)
            .mapKeys(taskRegistry::findTaskRecord)
            .flatMapKeys(Optional::stream)
            .mapKeys(TaskRegistry.Record::getErrorMeter)
            .forKeyValue((errorMeter, status) -> {
                errorMeter.set(status != TaskStatus.SUCCESS);
            });

        return Async.done();
    }

    private void heartbeat() {
        log.info("Heartbeat");

        updateStats()
            .whenComplete((v, e) -> {
                if (e != null) {
                    log.error("Stats update failed", e);
                }

                try {
                    val delay = workerConfiguration.getMonitorRate();
                    workerExecutor.schedule(this::heartbeat, delay.toMillis(), TimeUnit.MILLISECONDS);
                } catch (Exception ex) {
                    log.error("Can't schedule next heartbeat", ex);
                }
            });
    }

    @Override
    public void onApplicationEvent(StartupEvent event) {
        log.info("Monitor start");
        workerExecutor.submit(this::heartbeat);
    }
}
