package ru.yandex.mail.cerberus.worker.executer;

import io.micronaut.context.event.ApplicationEventListener;
import io.micronaut.context.event.StartupEvent;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import ru.yandex.mail.cerberus.IdempotencyKey;
import ru.yandex.mail.cerberus.dao.task.TaskRepository;
import ru.yandex.mail.cerberus.dao.task.TaskStatus;
import ru.yandex.mail.cerberus.dao.tx.TxManager;
import ru.yandex.mail.micronaut.common.qualifier.Master;
import ru.yandex.mail.cerberus.worker.TaskRegistry;
import ru.yandex.mail.cerberus.worker.WorkerConfiguration;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static ru.yandex.mail.cerberus.worker.executer.Worker.WORKER_EXECUTOR_NAME;

@Slf4j
@Singleton
public class Recycler implements ApplicationEventListener<StartupEvent> {
    private enum NextHeartbeatPolicy {
        EAGER,
        LAZY
    }

    private final ScheduledExecutorService workerExecutor;
    private final TaskRepository taskRepository;
    private final TxManager txManager;
    private final TaskRegistry taskRegistry;
    private final WorkerConfiguration workerConfiguration;

    @Inject
    public Recycler(@Named(WORKER_EXECUTOR_NAME) ExecutorService workerExecutor, TaskRepository taskRepository,
                    @Master TxManager txManager, TaskRegistry taskRegistry, WorkerConfiguration workerConfiguration) {
        this.workerExecutor = (ScheduledExecutorService) workerExecutor;
        this.taskRepository = taskRepository;
        this.txManager = txManager;
        this.taskRegistry = taskRegistry;
        this.workerConfiguration = workerConfiguration;
    }

    private CompletableFuture<NextHeartbeatPolicy> processExpiredTask() {
        return txManager.executeAsync(() -> {
            val taskOpt = taskRepository.acquireExpiredTask();
            if (taskOpt.isEmpty()) {
                log.info("Expired tasks not found");
                return NextHeartbeatPolicy.LAZY;
            }

            val task = taskOpt.get().getInfo();
            val type = task.getType();
            val key = task.getIdempotencyKey();
            log.info("{} task timeout exceeded, key = {}", type, key);

            taskRepository.finishTask(key, TaskStatus.TIMEOUT, Optional.of("Task timeout exceeded"));

            taskRegistry.findTaskConfiguration(type).ifPresent(config -> {
                config.getRepetitionRate().ifPresent(repetitionRate -> {
                    log.info("Task has cron nature, trying reschedule");
                    val newTask = taskRepository.insertTask(IdempotencyKey.random(), type, repetitionRate, config.getTimeout(),
                        task.getRequestId(), task.getInitiatorUid(), Optional.empty(), task.getContext());
                    log.info("Task rescheduled at '{}' with key '{}'", newTask.getSchedule(), newTask.getIdempotencyKey());
                });
            });

            return NextHeartbeatPolicy.EAGER;
        });
    }

    private void heartbeat() {
        log.info("Heartbeat");

        processExpiredTask()
            .whenComplete((policy, e) -> {
                if (e != null) {
                    log.error("Processing failed", e);
                    policy = NextHeartbeatPolicy.LAZY;
                }

                switch (policy) {
                    case EAGER:
                        scheduleNextHeartbeat(Duration.ZERO);
                        break;
                    case LAZY:
                        scheduleNextHeartbeat(workerConfiguration.getRecyclerRate());
                        break;
                }
            });
    }

    private void scheduleNextHeartbeat(Duration delay) {
        try {
            workerExecutor.schedule(this::heartbeat, delay.toMillis(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            log.error("Can't schedule next heartbeat", e);
        }
    }

    @Override
    public void onApplicationEvent(StartupEvent event) {
        log.info("Recycler start");
        workerExecutor.submit(this::heartbeat);
    }
}
