package ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync;

import java.time.OffsetDateTime;
import java.util.List;
import java.util.Optional;
import java.util.function.BinaryOperator;

import javax.inject.Inject;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import io.micronaut.core.annotation.Introspected;
import io.micronaut.core.order.Ordered;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.mail.cerberus.worker.api.Processor;
import ru.yandex.mail.cerberus.worker.api.TaskExecutionContext;
import ru.yandex.mail.cerberus.worker.api.TaskProcessor;
import ru.yandex.mail.cerberus.worker.api.exception.TaskTimeoutException;
import ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.sync.Synchronizer;
import ru.yandex.mail.micronaut.common.Async;

@Slf4j
@Processor(SyncStaffTaskProcessor.TYPE)
public class SyncStaffTaskProcessor implements TaskProcessor<SyncStaffTaskProcessor.Context> {
    static final String TYPE = "sync-staff";

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    @JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
    public static class Context {
        OffsetDateTime maxModifiedAt;
    }

    private final List<Synchronizer> synchronizers;

    @Inject
    public SyncStaffTaskProcessor(List<Synchronizer> synchronizers) {
        this.synchronizers = StreamEx.of(synchronizers)
            .sortedBy(Ordered::getOrder)
            .toImmutableList();
    }

    @Override
    public Class<Context> contextType() {
        return Context.class;
    }

    @Override
    public Mono<Context> process(Optional<Context> context, TaskExecutionContext executionContext) {
        final var syncPoint = context.map(Context::getMaxModifiedAt)
            .map(point -> point.minusDays(7));

        log.info("Start staff synchronization. Sync point = {}", syncPoint);
        val syncs = StreamEx.of(synchronizers)
            .map(synchronizer -> {
                return synchronizer.synchronize(executionContext, syncPoint)
                    .onErrorMap(Async::unwrap)
                    .doOnError(e -> log.error("'{}' failed", synchronizer.getName(), e))
                    .onErrorMap(e -> e instanceof TaskTimeoutException ? e : new SynchronizationFailedException(e));
            })
            .toImmutableList();

        return Flux.concat(syncs)
            .reduce(BinaryOperator.maxBy(OffsetDateTime.timeLineOrder()))
            .map(Context::new)
            .doOnSuccess(ignore -> log.info("Staff synchronization complete successfully"));
    }
}
