package ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.sync;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.slf4j.Logger;
import reactor.core.publisher.Flux;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.client.dto.Group;
import ru.yandex.mail.cerberus.core.group.GroupManager;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.SyncStaffTaskConfiguration;
import ru.yandex.mail.cerberus.yt.data.YtDepartmentInfo;
import ru.yandex.mail.cerberus.yt.mapper.YtDepartmentMapper;
import ru.yandex.mail.cerberus.yt.staff.StaffManager;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffDepartmentGroup;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_DEPARTMENT_GROUP_TYPE;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@Singleton
@Slf4j(topic = "group-sync")
class GroupSyncProvider implements SyncProvider<GroupId, BasicContext, Group<YtDepartmentInfo>, StaffDepartmentGroup> {
    private final StaffManager staffManager;
    private final GroupManager groupManager;
    private final YtDepartmentMapper mapper;
    private final int chunkSize;

    @Inject
    public GroupSyncProvider(StaffManager staffManager, GroupManager groupManager, YtDepartmentMapper mapper,
                             SyncStaffTaskConfiguration configuration) {
        this.staffManager = staffManager;
        this.groupManager = groupManager;
        this.mapper = mapper;
        this.chunkSize = configuration.getGroupChunkSize();
    }

    @Override
    public int getMaxChunkSize() {
        return chunkSize;
    }

    @Override
    public String getSyncEntityName() {
        return "department";
    }

    @Override
    public Logger getLog() {
        return log;
    }

    @Override
    public SyncOrder getOrder() {
        return SyncOrder.GROUP;
    }

    @Override
    public OffsetDateTime getModifiedAt(StaffDepartmentGroup group) {
        return group.getMeta().getModifiedAt();
    }

    @Override
    public GroupId getIdForDto(Group<YtDepartmentInfo> group) {
        return group.getId();
    }

    @Override
    public GroupId getIdForStaffDto(StaffDepartmentGroup group) {
        return group.getId();
    }

    @Override
    public SyncDecision isReadyToSync(StaffDepartmentGroup group, BasicContext context) {
        if (group.isDeleted()) {
            return new SyncDecision.UpdateOnly("group is deleted");
        } else {
            return SyncDecision.SYNC;
        }
    }

    @Override
    public Group<YtDepartmentInfo> update(Group<YtDepartmentInfo> group, StaffDepartmentGroup department, BasicContext context) {
        return mapper.mapToGroup(department);
    }

    @Override
    public Flux<Batch<StaffDepartmentGroup, BasicContext>> batches(Optional<OffsetDateTime> syncPoint) {
        return staffManager.departmentsRx(chunkSize, syncPoint)
            .map(elements -> new Batch<>(elements, BasicContext.empty()));
    }

    @Override
    public CompletableFuture<List<Group<YtDepartmentInfo>>> findExisting(List<StaffDepartmentGroup> groups, BasicContext context) {
        val ids = mapToSet(groups, StaffDepartmentGroup::getId);
        return groupManager.findGroups(YtDepartmentInfo.class, YT_DEPARTMENT_GROUP_TYPE, ids, ReadTarget.MASTER);
    }

    @Override
    public CompletableFuture<List<Group<YtDepartmentInfo>>> commitNew(List<StaffDepartmentGroup> staffGroups, BasicContext context) {
        val groups = mapToList(staffGroups, mapper::mapToGroup);
        return groupManager.insert(groups);
    }

    @Override
    public CompletableFuture<Void> commitChanged(List<Group<YtDepartmentInfo>> groups, BasicContext context) {
        return groupManager.update(groups);
    }
}
