package ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.sync;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.slf4j.Logger;
import reactor.core.publisher.Flux;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.client.dto.Location;
import ru.yandex.mail.cerberus.core.location.LocationManager;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.SyncStaffTaskConfiguration;
import ru.yandex.mail.cerberus.yt.data.YtOfficeInfo;
import ru.yandex.mail.cerberus.yt.mapper.YtOfficeMapper;
import ru.yandex.mail.cerberus.yt.staff.StaffManager;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffOffice;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_OFFICE_TYPE;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@Singleton
@Slf4j(topic = "office-sync")
class OfficeSyncProvider implements SyncProvider<LocationId, BasicContext, Location<YtOfficeInfo>, StaffOffice> {
    private final StaffManager staffManager;
    private final LocationManager locationManager;
    private final YtOfficeMapper mapper;
    private final int chunkSize;

    @Inject
    public OfficeSyncProvider(StaffManager staffManager, LocationManager locationManager, YtOfficeMapper mapper,
                              SyncStaffTaskConfiguration configuration) {
        this.staffManager = staffManager;
        this.locationManager = locationManager;
        this.mapper = mapper;
        this.chunkSize = configuration.getOfficeChunkSize();
    }

    @Override
    public int getMaxChunkSize() {
        return chunkSize;
    }

    @Override
    public String getSyncEntityName() {
        return "office";
    }

    @Override
    public Logger getLog() {
        return log;
    }

    @Override
    public SyncOrder getOrder() {
        return SyncOrder.OFFICE;
    }

    @Override
    public OffsetDateTime getModifiedAt(StaffOffice office) {
        return office.getMeta().getModifiedAt();
    }

    @Override
    public LocationId getIdForDto(Location<YtOfficeInfo> office) {
        return office.getId();
    }

    @Override
    public LocationId getIdForStaffDto(StaffOffice office) {
        return office.getId();
    }

    @Override
    public SyncDecision isReadyToSync(StaffOffice office, BasicContext context) {
        if (office.isDeleted()) {
            return new SyncDecision.UpdateOnly("office is deleted");
        } else {
            return SyncDecision.SYNC;
        }
    }

    @Override
    public Location<YtOfficeInfo> update(Location<YtOfficeInfo> office, StaffOffice staffOffice, BasicContext context) {
        return mapper.mapToLocation(staffOffice);
    }

    @Override
    public Flux<Batch<StaffOffice, BasicContext>> batches(Optional<OffsetDateTime> syncPoint) {
        return staffManager.officesRx(chunkSize, syncPoint)
            .map(chunk -> new Batch<>(chunk, BasicContext.empty()));
    }

    @Override
    public CompletableFuture<List<Location<YtOfficeInfo>>> findExisting(List<StaffOffice> offices, BasicContext context) {
        val ids = mapToSet(offices, StaffOffice::getId);
        return locationManager.findByType(YtOfficeInfo.class, YT_OFFICE_TYPE, ids, ReadTarget.MASTER);
    }

    @Override
    public CompletableFuture<List<Location<YtOfficeInfo>>> commitNew(List<StaffOffice> staffOffices, BasicContext context) {
        val offices = mapToList(staffOffices, mapper::mapToLocation);
        return locationManager.insert(offices);
    }

    @Override
    public CompletableFuture<Void> commitChanged(List<Location<YtOfficeInfo>> offices, BasicContext context) {
        return locationManager.update(offices);
    }
}
