package ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.sync;

import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.ResourceId;
import ru.yandex.mail.cerberus.client.dto.Location;
import ru.yandex.mail.cerberus.client.dto.Resource;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.location.LocationManager;
import ru.yandex.mail.cerberus.core.resource.ResourceManager;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.SyncStaffTaskConfiguration;
import ru.yandex.mail.cerberus.yt.data.YtRoomInfo;
import ru.yandex.mail.cerberus.yt.mapper.YtRoomMapper;
import ru.yandex.mail.cerberus.yt.staff.StaffEntity;
import ru.yandex.mail.cerberus.yt.staff.StaffManager;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffRoom;
import ru.yandex.mail.micronaut.common.Async;
import ru.yandex.mail.micronaut.common.RawJsonString;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import static com.ea.async.Async.await;
import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_OFFICE_TYPE;
import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_ROOM_RESOURCE_TYPE_NAME;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@Singleton
@Slf4j(topic = "room-sync")
class RoomSyncProvider implements SyncProvider<ResourceId, RoomSyncProvider.Context, Resource<YtRoomInfo>, StaffRoom> {
    @Value
    public static final class Context {
        Set<LocationId> existingOffices;
    }

    private static final Predicate<String> EXCHANGE_NAME_PATTERN = Pattern.compile("[a-zA-Z0-9_][.a-zA-Z0-9_-]*").asMatchPredicate();

    private final ResourceManager resourceManager;
    private final StaffManager staffManager;
    private final YtRoomMapper mapper;
    private final LocationManager locationManager;
    private final int chunkSize;

    @Inject
    public RoomSyncProvider(ResourceManager resourceManager, StaffManager staffManager, YtRoomMapper mapper,
                            LocationManager locationManager, SyncStaffTaskConfiguration configuration) {
        this.resourceManager = resourceManager;
        this.staffManager = staffManager;
        this.mapper = mapper;
        this.locationManager = locationManager;
        this.chunkSize = configuration.getRoomChunkSize();
    }

    @Override
    public int getMaxChunkSize() {
        return chunkSize;
    }

    @Override
    public String getSyncEntityName() {
        return "room";
    }

    @Override
    public Logger getLog() {
        return log;
    }

    @Override
    public SyncOrder getOrder() {
        return SyncOrder.ROOM;
    }

    @Override
    public OffsetDateTime getModifiedAt(StaffRoom room) {
        return room.getMeta().getModifiedAt();
    }

    @Override
    public ResourceId getIdForDto(Resource<YtRoomInfo> room) {
        return room.getId();
    }

    @Override
    public ResourceId getIdForStaffDto(StaffRoom room) {
        return room.getId();
    }

    private static boolean isValidExchangeName(String exchangeName) {
        return EXCHANGE_NAME_PATTERN.test(exchangeName);
    }

    private StreamEx<String> packReasonIf(boolean condition, String reason) {
        return condition ? StreamEx.of(reason) : StreamEx.empty();
    }

    @Override
    public SyncDecision isReadyToSync(StaffRoom room, Context context) {
        val exchangeName = room.getName().getExchange();
        val officeId = room.getFloor().getOffice().getId();

        val hasValidExchangeName = isValidExchangeName(exchangeName);
        val hasValidOffice = context.getExistingOffices().contains(officeId);

        val reasons = StreamEx.<String>empty()
            .append(packReasonIf(room.isDeleted(), "room is deleted"))
            .append(packReasonIf(!hasValidExchangeName, "room has invalid exchange name " + exchangeName))
            .append(packReasonIf(!hasValidOffice, "room has invalid office reference " + officeId))
            .toImmutableSet();

        if (reasons.isEmpty()) {
            return SyncDecision.SYNC;
        } else {
            return hasValidOffice ? new SyncDecision.UpdateOnly(reasons) : new SyncDecision.DontSync(reasons);
        }
    }

    @Override
    public Resource<YtRoomInfo> update(Resource<YtRoomInfo> room, StaffRoom staffRoom, Context context) {
        val newRoom = mapper.mapToResource(staffRoom);

        val oldName = room.getName();
        val newName = newRoom.getName();
        if (!oldName.equals(newName)) {
            log.warn("New name={}, old name={}, but we always keep the old one", newName, oldName);
            val newData = newRoom.getData().withName(oldName);
            return new Resource<>(newRoom.getId(), newData);
        } else {
            return newRoom;
        }
    }

    private CompletableFuture<Batch<StaffRoom, Context>> fetchBatch(List<StaffEntity<StaffRoom>> chunk) {
        val officeIds = StreamEx.of(chunk)
            .filter(StaffEntity::isValid)
            .map(StaffEntity::getEntity)
            .map(room -> room.getFloor().getOffice().getId())
            .toImmutableSet();

        val offices = await(locationManager.findByType(RawJsonString.class, YT_OFFICE_TYPE, officeIds, ReadTarget.MASTER));
        log.info("{} existing offices found", offices.size());

        val existingOfficeIds = mapToSet(offices, Location::getId);
        val batch = new Batch<>(chunk, new Context(existingOfficeIds));
        return Async.done(batch);
    }

    @Override
    public Flux<Batch<StaffRoom, Context>> batches(Optional<OffsetDateTime> syncPoint) {
        return staffManager.meetingRoomsRx(chunkSize, syncPoint)
            .flatMap(chunk -> Mono.fromFuture(fetchBatch(chunk)), 1, 1);
    }

    @Override
    public CompletableFuture<List<Resource<YtRoomInfo>>> findExisting(List<StaffRoom> rooms, Context context) {
        val ids = mapToSet(rooms, StaffRoom::getId);
        return resourceManager.findResources(YT_ROOM_RESOURCE_TYPE_NAME, ids, YtRoomInfo.class, ReadTarget.MASTER);
    }

    @Override
    public CompletableFuture<List<Resource<YtRoomInfo>>> commitNew(List<StaffRoom> staffRooms, Context context) {
        val rooms = mapToList(staffRooms, mapper::mapToResource);
        return resourceManager.insertResources(CollisionStrategy.FAIL, rooms);
    }

    @Override
    public CompletableFuture<Void> commitChanged(List<Resource<YtRoomInfo>> rooms, Context context) {
        return resourceManager.updateResources(rooms);
    }
}
