package ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.sync;

import io.micronaut.core.annotation.Introspected;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.cerberus.core.CollisionStrategy;
import ru.yandex.mail.cerberus.core.group.GroupManager;
import ru.yandex.mail.cerberus.core.user.UserManager;
import ru.yandex.mail.cerberus.ReadTarget;
import ru.yandex.mail.cerberus.worker.yt_tasks.staff_sync.SyncStaffTaskConfiguration;
import ru.yandex.mail.cerberus.yt.data.YtUserInfo;
import ru.yandex.mail.cerberus.yt.mapper.YtUserMapper;
import ru.yandex.mail.cerberus.yt.staff.StaffEntity;
import ru.yandex.mail.cerberus.yt.staff.StaffManager;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.mail.micronaut.common.Async;
import ru.yandex.mail.micronaut.common.CerberusUtils;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.ea.async.Async.await;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_DEPARTMENT_GROUP_TYPE;
import static ru.yandex.mail.micronaut.common.Async.runIf;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToMap;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToSet;

@Value
@Introspected
class UserWithGroups {
    private final User<YtUserInfo> user;
    private final Set<GroupId> groups;
}

@Singleton
@Slf4j(topic = "user-sync")
class UserSyncProvider implements SyncProvider<Uid, UserSyncProvider.Context, UserWithGroups, StaffUser> {
    @Value
    public static class Context {
        Map<Uid, Set<GroupId>> userDepartments;
    }

    private final StaffManager staffManager;
    private final UserManager userManager;
    private final GroupManager groupManager;
    private final YtUserMapper mapper;
    private final int chunkSize;

    @Inject
    public UserSyncProvider(StaffManager staffManager, UserManager userManager, GroupManager groupManager,
                            YtUserMapper mapper, SyncStaffTaskConfiguration configuration) {
        this.staffManager = staffManager;
        this.userManager = userManager;
        this.groupManager = groupManager;
        this.mapper = mapper;
        this.chunkSize = configuration.getUserChunkSize();
    }

    @Override
    public int getMaxChunkSize() {
        return chunkSize;
    }

    @Override
    public String getSyncEntityName() {
        return "user";
    }

    @Override
    public Logger getLog() {
        return log;
    }

    @Override
    public SyncOrder getOrder() {
        return SyncOrder.USER;
    }

    @Override
    public OffsetDateTime getModifiedAt(StaffUser user) {
        return user.getMeta().getModifiedAt();
    }

    @Override
    public Uid getIdForDto(UserWithGroups user) {
        return user.getUser().getUid();
    }

    @Override
    public Uid getIdForStaffDto(StaffUser user) {
        return user.safeUid();
    }

    @Override
    public SyncDecision isReadyToSync(StaffUser user, Context context) {
        if (user.isDeleted()) {
            return new SyncDecision.UpdateOnly("user is deleted");
        } else {
            return SyncDecision.SYNC;
        }
    }

    @Override
    public UserWithGroups update(UserWithGroups userWithGroups, StaffUser staffUser, Context context) {
        val newUser = mapper.mapToUser(staffUser);
        val newGroups = staffUser.departmentIds().toImmutableSet();

        val newLogin = newUser.getLogin();
        val oldLogin = userWithGroups.getUser().getLogin();
        if (!newLogin.equals(oldLogin)) {
            log.warn("New login={}, old login={}, but we always keep the old one", newLogin, oldLogin);
            return new UserWithGroups(newUser.withLogin(oldLogin), newGroups);
        } else {
            return new UserWithGroups(newUser, newGroups);
        }
    }

    private CompletableFuture<Batch<StaffUser, Context>> fetchBatch(List<StaffEntity<StaffUser>> chunk) {
        val uids = StreamEx.of(chunk)
            .filter(StaffEntity::isValid)
            .map(StaffEntity::getEntity)
            .map(StaffUser::safeUid)
            .toImmutableSet();
        val userDepartments = await(userManager.findUsersGroupsByType(uids, YT_DEPARTMENT_GROUP_TYPE, ReadTarget.MASTER));
        val batch = new Batch<>(chunk, new Context(userDepartments));
        return Async.done(batch);
    }

    @Override
    public Flux<Batch<StaffUser, Context>> batches(Optional<OffsetDateTime> syncPoint) {
        return staffManager.usersRx(chunkSize, syncPoint)
            .flatMap(chunk -> Mono.fromFuture(fetchBatch(chunk)), 1, 1);
    }

    @Override
    public CompletableFuture<List<UserWithGroups>> findExisting(List<StaffUser> users, Context context) {
        val uids = mapToSet(users, StaffUser::safeUid);
        val existingUsers = await(userManager.find(uids, YtUserInfo.class, ReadTarget.MASTER));

        val existingUsersMap = mapToMap(existingUsers, User::getUid, identity());

        val usersWithGroups = EntryStream.of(context.getUserDepartments())
            .mapKeyValue((uid, groupsSet) -> new UserWithGroups(existingUsersMap.get(uid), groupsSet))
            .toImmutableList();
        return Async.done(usersWithGroups);
    }

    @Override
    public CompletableFuture<List<UserWithGroups>> commitNew(List<StaffUser> users, Context context) {
        final var usersWithGroups = StreamEx.of(users)
            .map(user -> {
                val groups = user.departmentIds().toImmutableSet();
                return new UserWithGroups(mapper.mapToUser(user), groups);
            })
            .toImmutableList();

        final var groupIdsByUid = StreamEx.of(usersWithGroups)
            .toMap(info -> info.getUser().getUid(), UserWithGroups::getGroups);

        await(userManager.insert(CollisionStrategy.FAIL, mapToList(usersWithGroups, UserWithGroups::getUser)));
        await(groupManager.addUsers(YT_DEPARTMENT_GROUP_TYPE, groupIdsByUid));
        return Async.done(usersWithGroups);
    }

    private enum MappingDirection {
        ADDED,
        REMOVED
    }

    private static Map<Uid, Set<GroupId>> resolveMapping(MappingDirection direction,
                                                         EntryStream<Uid, Set<GroupId>> newState,
                                                         Map<Uid, Set<GroupId>> oldState) {
        return newState.mapToValue((uid, newGroupSet) -> {
            val oldGroupSet = oldState.getOrDefault(uid, emptySet());
            if (direction == MappingDirection.ADDED) {
                return CerberusUtils.findMissing(oldGroupSet, newGroupSet, Collectors.toSet());
            } else {
                return CerberusUtils.findMissing(newGroupSet, oldGroupSet, Collectors.toSet());
            }
        })
        .removeValues(Set::isEmpty)
        .toImmutableMap();
    }

    private static EntryStream<Uid, Set<GroupId>> toStream(List<UserWithGroups> users) {
        return StreamEx.of(users)
            .mapToEntry(
                userWithGroups -> userWithGroups.getUser().getUid(),
                UserWithGroups::getGroups
            );
    }

    @Override
    public CompletableFuture<Void> commitChanged(List<UserWithGroups> users, Context context) {
        val additionMapping = resolveMapping(MappingDirection.ADDED, toStream(users), context.getUserDepartments());
        val removingMapping = resolveMapping(MappingDirection.REMOVED, toStream(users), context.getUserDepartments());

        await(userManager.update(mapToList(users, UserWithGroups::getUser)));

        return CompletableFuture.allOf(
            runIf(!removingMapping.isEmpty(), () -> groupManager.removeUsers(YT_DEPARTMENT_GROUP_TYPE, removingMapping)),
            runIf(!additionMapping.isEmpty(), () -> groupManager.addUsers(YT_DEPARTMENT_GROUP_TYPE, additionMapping))
        );
    }
}
