package ru.yandex.mail.cerberus.yt.staff;

import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.inject.Inject;
import javax.inject.Singleton;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import lombok.SneakyThrows;
import lombok.Value;
import lombok.val;
import one.util.streamex.MoreCollectors;
import one.util.streamex.StreamEx;

import ru.yandex.mail.cerberus.yt.staff.client.StaffClient;
import ru.yandex.mail.cerberus.yt.staff.client.StaffResult;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffDepartmentGroup;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffDto;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffOffice;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffRoom;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.mail.micronaut.common.Page;

import static com.ea.async.Async.await;
import static ru.yandex.mail.micronaut.common.Async.done;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
class StaffEntityId implements StaffDto {
    long id;

    @Override
    @JsonIgnore
    public long getUniqueId() {
        return id;
    }
}

@Singleton
public class DefaultStaffManager implements StaffManager {
    @FunctionalInterface
    private interface ItemConverter<T> {
        StaffEntity<T> convert(JsonNode node);
    }

    @FunctionalInterface
    interface PageFetcher {
        CompletableFuture<StaffResult> fetch();
    }

    private static final String ROOM_FIELDS = "id,is_deleted,is_bookable,type,name,floor.id,floor.number,floor.office.id,"
            + "additional,capacity,phone,equipment.cork_board,equipment.desk,"
            + "equipment.game_console,equipment.guest_wifi,equipment.projector,"
            + "equipment.screen,equipment.seats,equipment.video_conferencing,"
            + "equipment.voice_conferencing,equipment.marker_board,_meta.modified_at";
    private static final String USER_FIELDS = "id,uid,login,is_deleted,official.is_dismissed,official.is_robot,"
            + "official.is_homeworker,official.affiliation,department_group.id,"
            + "department_group.is_deleted,department_group.ancestors.id,"
            + "department_group.ancestors.is_deleted,work_email,language.ui,"
            + "groups.group.id,groups.group.url,groups.group.is_deleted,"
            + "groups.group.ancestors.id,groups.group.ancestors.is_deleted,"
            + "environment.timezone,location.office.id,location.table.floor.number,"
            + "name.first.ru,name.last.ru,name.first.en,name.last.en,official.position,"
            + "name.middle,personal.gender,work_phone,phones.type,phones.number,"
            + "phones.is_main,cars,_meta.modified_at";
    private static final String DEPARTMENT_FIELDS = "id,is_deleted,url,name,type,"
            + "department.is_deleted,department.name.full,"
            + "department.id,department.url,department.heads.person.uid,"
            + "ancestors.is_deleted,ancestors.department.id,"
            + "ancestors.department.level,ancestors.department.heads.person.uid,"
            + "_meta.modified_at";
    private static final String OFFICE_FIELDS = "id,name.en,name.ru,code,city.name.en,city.name.ru,timezone,is_deleted,"
            + "_meta.modified_at";

    private final StaffClient staffClient;
    private final ObjectReader roomReader;
    private final ObjectReader userReader;
    private final ObjectReader departmentGroupReader;
    private final ObjectReader officeReader;
    private final ObjectReader entityIdReader;

    @Inject
    public DefaultStaffManager(StaffClient staffClient, ObjectMapper objectMapper) {
        this.staffClient = staffClient;
        roomReader = objectMapper.readerFor(StaffRoom.class);
        userReader = objectMapper.readerFor(StaffUser.class);
        departmentGroupReader = objectMapper.readerFor(StaffDepartmentGroup.class);
        officeReader = objectMapper.readerFor(StaffOffice.class);
        entityIdReader = objectMapper.readerFor(StaffEntityId.class);
    }

    private static String toQuery(long pageId, Optional<OffsetDateTime> modifiedSince) {
        final var modifiedAtFilter = modifiedSince
                .map(time -> {
                    val utcTime = time.withOffsetSameInstant(ZoneOffset.UTC)
                            .format(DateTimeFormatter.ISO_OFFSET_DATE_TIME);
                    return "_meta.modified_at>'" + utcTime + '\'';
                });

        return StreamEx.of("id>" + pageId)
                .append(modifiedAtFilter.stream())
                .joining(" and ");
    }

    static <T extends StaffDto> CompletableFuture<Page<Long, StaffEntity<T>>> doRequest(int pageSize,
                                                                                        PageFetcher pageFetcher,
                                                                                        ItemConverter<T> converter) {
        if (pageSize < 1) {
            throw new IllegalArgumentException("Page size needs to be > 0");
        }

        val staffResult = await(pageFetcher.fetch());
        val elements = mapToList(staffResult.getResult(), converter::convert);
        val hasNext = (elements.size() == pageSize);
        val lastElementId = StreamEx.of(elements)
                .map(StaffEntity::getUniqueId)
                .flatMap(StreamEx::of)
                .collect(MoreCollectors.last());

        val nextPageId = hasNext ? lastElementId : Optional.<Long>empty();
        return done(new Page<>(elements, nextPageId));
    }

    @SneakyThrows
    private static <T> T parseObject(ObjectReader reader, JsonNode json) {
        return reader.readValue(json);
    }

    private <T extends StaffDto> ItemConverter<T> asConverter(Function<JsonNode, T> parser) {
        return asConverter(parser, entityIdReader);
    }

    static  <T extends StaffDto> ItemConverter<T> asConverter(Function<JsonNode, T> parser, ObjectReader entityIdReader) {
        return node -> {
            try {
                return new StaffEntity.Valid<>(parser.apply(node));
            } catch (Exception ignore) {
                try {
                    final StaffEntityId entityId = parseObject(entityIdReader, node);
                    return new StaffEntity.Invalid<>(node, Optional.of(entityId.getId()));
                } catch (Exception ignore2) {
                    return new StaffEntity.Invalid<>(node, Optional.empty());
                }
            }
        };
    }

    private StaffRoom parseRoom(JsonNode json) {
        return parseObject(roomReader, json);
    }

    private StaffUser parseUser(JsonNode json) {
        return parseObject(userReader, json);
    }

    private StaffDepartmentGroup parseDepartment(JsonNode json) {
        return parseObject(departmentGroupReader, json);
    }

    private StaffOffice parseOffice(JsonNode json) {
        return parseObject(officeReader, json);
    }

    @Override
    public CompletableFuture<Page<Long, StaffEntity<StaffRoom>>> meetingRooms(long pageId, int pageSize,
                                                                              Optional<OffsetDateTime> modifiedSince) {
        val query = toQuery(pageId, modifiedSince);
        return doRequest(pageSize, () -> staffClient.rooms(1, "id", pageSize, query, StaffRoom.Type.CONFERENCE, ROOM_FIELDS),
                asConverter(this::parseRoom));
    }

    @Override
    public CompletableFuture<Page<Long, StaffEntity<StaffUser>>> users(long pageId, int pageSize,
                                                                       Optional<OffsetDateTime> modifiedSince) {
        val query = toQuery(pageId, modifiedSince);
        return doRequest(pageSize, () -> staffClient.persons(1, "id", pageSize, query, USER_FIELDS),
                asConverter(this::parseUser));
    }

    @Override
    public CompletableFuture<Page<Long, StaffEntity<StaffDepartmentGroup>>> departments(long pageId, int pageSize,
                                                                                        Optional<OffsetDateTime> modifiedSince) {
        val query = toQuery(pageId, modifiedSince);
        return doRequest(pageSize, () -> staffClient.groups(1, "id", pageSize, query, DEPARTMENT_FIELDS),
                asConverter(this::parseDepartment));
    }

    @Override
    public CompletableFuture<Page<Long, StaffEntity<StaffOffice>>> offices(long pageId, int pageSize,
                                                                           Optional<OffsetDateTime> modifiedSince) {
        val query = toQuery(pageId, modifiedSince);
        return doRequest(pageSize, () -> staffClient.offices(1, "id", pageSize, query, OFFICE_FIELDS),
                asConverter(this::parseOffice));
    }
}
