package ru.yandex.mail.cerberus.yt.staff;

import reactor.core.publisher.Flux;
import ru.yandex.mail.micronaut.common.Async;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffDepartmentGroup;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffOffice;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffRoom;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.mail.micronaut.common.Page;
import ru.yandex.mail.micronaut.common.Pageable;

import java.time.OffsetDateTime;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

@FunctionalInterface
interface StaffPageFetcher<T> {
    CompletableFuture<Page<Long, StaffEntity<T>>> fetch(long pageId, int pageSize, Optional<OffsetDateTime> modifiedSince);
}

public interface StaffManager {
    long START_PAGE_ID = -1;

    private static <T> Flux<List<StaffEntity<T>>> fetchRx(int chunkSize, Optional<OffsetDateTime> modifiedSince,
                                                          StaffPageFetcher<T> fetcher) {
        return Async.fetchPagesRx(chunkSize, (Pageable<Long> pageable) -> {
            return fetcher.fetch(pageable.getPageId().orElse(START_PAGE_ID), pageable.getPageSize(), modifiedSince);
        });
    }

    CompletableFuture<Page<Long, StaffEntity<StaffRoom>>> meetingRooms(long pageId, int pageSize,
                                                                       Optional<OffsetDateTime> modifiedSince);

    default Flux<List<StaffEntity<StaffRoom>>> meetingRoomsRx(int chunkSize, Optional<OffsetDateTime> modifiedSince) {
        return fetchRx(chunkSize, modifiedSince, this::meetingRooms);
    }

    CompletableFuture<Page<Long, StaffEntity<StaffUser>>> users(long pageId, int pageSize,
                                                                Optional<OffsetDateTime> modifiedSince);

    default Flux<List<StaffEntity<StaffUser>>> usersRx(int chunkSize, Optional<OffsetDateTime> modifiedSince) {
        return fetchRx(chunkSize, modifiedSince, this::users);
    }

    CompletableFuture<Page<Long, StaffEntity<StaffDepartmentGroup>>> departments(long pageId, int pageSize,
                                                                                 Optional<OffsetDateTime> modifiedSince);

    default Flux<List<StaffEntity<StaffDepartmentGroup>>> departmentsRx(int chunkSize, Optional<OffsetDateTime> modifiedSince) {
        return fetchRx(chunkSize, modifiedSince, this::departments);
    }

    CompletableFuture<Page<Long, StaffEntity<StaffOffice>>> offices(long pageId, int pageSize,
                                                                    Optional<OffsetDateTime> modifiedSince);

    default Flux<List<StaffEntity<StaffOffice>>> officesRx(int chunkSize, Optional<OffsetDateTime> modifiedSince) {
        return fetchRx(chunkSize, modifiedSince, this::offices);
    }
}
