package ru.yandex.mail.cerberus.yt.staff.dto;

import java.time.ZoneId;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;

import javax.validation.constraints.Email;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.PositiveOrZero;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import io.micronaut.core.annotation.Introspected;
import lombok.AllArgsConstructor;
import lombok.Value;
import one.util.streamex.StreamEx;

import ru.yandex.mail.cerberus.GroupId;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.mail.cerberus.Uid;

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
public class StaffUser implements StaffDto {
    public enum Affiliation {
        YANDEX,
        YAMONEY,
        EXTERNAL;

        @Override
        public String toString() {
            return super.toString().toLowerCase();
        }
    }

    public enum PhoneType {
        MOBILE,
        HOME;

        @Override
        public String toString() {
            return super.toString().toLowerCase();
        }
    }

    public enum Gender {
        MALE,
        FEMALE;

        @Override
        public String toString() {
            return super.toString().toLowerCase();
        }
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Official {
        @JsonProperty("is_dismissed") boolean dismissed;
        @JsonProperty("is_robot") boolean robot;
        @JsonProperty("is_homeworker") boolean homeworker;
        Affiliation affiliation;
        StaffLocalizedString position;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Personal {
        Gender gender;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class DepartmentAncestor {
        GroupId id;
        @JsonProperty("is_deleted") boolean deleted;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class DepartmentGroup {
        GroupId id;
        List<DepartmentAncestor> ancestors;
        @JsonProperty("is_deleted") boolean deleted;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Group {
        InnerGroup group;

        @Value
        @Introspected
        @AllArgsConstructor(onConstructor_= @JsonCreator)
        public static class InnerGroup {
            GroupId id;
            @JsonProperty("url")
            String url;
            @JsonProperty("is_deleted")
            boolean deleted;
            List<GroupAncestors> ancestors;

            @Value
            @Introspected
            @AllArgsConstructor(onConstructor_ = @JsonCreator)
            public static class GroupAncestors {
                GroupId id;
                @JsonProperty("is_deleted")
                boolean deleted;
            }
        }
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Environment {
        ZoneId timezone;
    }

    @Value
    @Introspected
    public static class Location {
        @Value
        @Introspected
        @AllArgsConstructor(onConstructor_= @JsonCreator)
        public static class Office {
            LocationId id;
        }

        @Value
        @Introspected
        @AllArgsConstructor(onConstructor_= @JsonCreator)
        public static class Table {
            @Value
            @Introspected
            @AllArgsConstructor(onConstructor_= @JsonCreator)
            public static class Floor {
                OptionalInt number;
            }

            Floor floor;
        }

        Office office;
        Table table;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Language {
        String ui;
    }

    @Value
    @Introspected
    public static class Name {
        StaffLocalizedString first;
        StaffLocalizedString last;
        Optional<String> middle;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_= @JsonCreator)
    public static class Phone {
        PhoneType type;
        String number;
        @JsonProperty("is_main") boolean main;
    }

    @Value
    @Introspected
    @AllArgsConstructor(onConstructor_=@JsonCreator)
    public static class Car {
        String plate;
        String model;
    }

    @JsonIgnore
    public StreamEx<GroupId> departmentIds() {
        Set<GroupId> result = new HashSet<>();
        for (DepartmentAncestor ancestor : departmentGroup.ancestors) {
            if (!ancestor.deleted) {
                result.add(ancestor.id);
            }
        }
        userDepartmentId().ifPresent(result::add);
        for (Group group : groups) {
            if (!group.group.deleted) {
                result.add(group.group.id);
                for (Group.InnerGroup.GroupAncestors ancestor : group.group.getAncestors()) {
                    if (!ancestor.deleted) {
                        result.add(ancestor.id);
                    }
                }
            }
        }
        return StreamEx.of(result);
    }

    @JsonIgnore
    public Optional<GroupId> userDepartmentId() {
        return departmentGroup.isDeleted() ? Optional.empty() : Optional.of(departmentGroup.getId());
    }

    @JsonIgnore
    public Uid safeUid() {
        return new Uid(Long.parseLong(uid));
    }

    @Override
    @JsonIgnore
    public long getUniqueId() {
        return id;
    }

    @JsonProperty("_meta") Meta meta;
    @PositiveOrZero long id;
    @NotBlank String uid;
    @NotBlank String login;
    @JsonProperty("is_deleted") boolean deleted;
    Official official;
    Personal personal;
    DepartmentGroup departmentGroup;
    List<Group> groups;
    @Email String workEmail;
    Optional<@PositiveOrZero Long> workPhone;
    List<Phone> phones;
    Environment environment;
    Location location;
    Language language;
    Name name;
    List<Car> cars;
    Optional<String> workMode;
}
