from typing import Any

from aiohttp import web

from sendr_aiohttp import BaseUrlDispatcher
from sendr_qlog.http.aiohttp import get_middleware_logging_adapter, signal_request_id_header
from sendr_taskqueue.background import BackgroundSchedule

from mail.ciao.ciao import __version__
from mail.ciao.ciao.api.middlewares import (
    middleware_logging_context, middleware_response_formatter, middleware_stats, middleware_tvm_restrictor
)
from mail.ciao.ciao.api.pinger import PingerTask
from mail.ciao.ciao.api.routes.megamind import MEGAMIND_ROUTES
from mail.ciao.ciao.api.routes.utility import UTILITY_ROUTES
from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.interactions.base import BaseInteractionClient, create_connector


class CiaoApplication(web.Application):  # type: ignore
    _urls = (
        MEGAMIND_ROUTES,
        UTILITY_ROUTES,
    )

    def __init__(self):
        super().__init__(
            router=BaseUrlDispatcher(),
            middlewares=(
                middleware_stats,
                get_middleware_logging_adapter(),
                middleware_logging_context,
                middleware_response_formatter,
                middleware_tvm_restrictor,
            ),
        )
        self.add_routes()
        self.add_sentry()
        self.add_schedule()
        self.on_response_prepare.append(signal_request_id_header)
        self.on_startup.append(self.create_connector)
        self.on_cleanup.append(self.close_connector)

    def add_routes(self) -> None:  # type: ignore
        for routes in self._urls:
            self.router.add_routes(routes)

    def add_sentry(self) -> None:
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))

    def add_schedule(self) -> None:
        schedule = BackgroundSchedule(PingerTask())
        self.on_startup.append(schedule.run)
        self.on_cleanup.append(schedule.stop)

    async def create_connector(self, _: Any) -> None:
        BaseInteractionClient.CONNECTOR = create_connector()

    async def close_connector(self, _: Any) -> None:
        await BaseInteractionClient.close_connector()
