from marshmallow import SchemaOpts, post_dump, pre_load

from google.protobuf import json_format
from mail.ciao.ciao.api.schemas.base import BaseSchema


class ProtobufOpts(SchemaOpts):
    def __init__(self, meta):
        super().__init__(meta)
        self.protobuf = getattr(meta, 'protobuf', None)


class BaseMegamindSchema(BaseSchema):
    OPTIONS_CLASS = ProtobufOpts

    @property
    def protobuf_context(self) -> bool:
        return self.context.get('protobuf')

    @property
    def protobuf_data(self) -> bool:
        # Every schema has protobuf context and can use it to change behaviour, but only top-most schemas have actual
        # protobuf to use for dumping / loading.
        return self.protobuf_context and self.opts.protobuf

    @pre_load
    def protobuf_to_json(self, data):
        if not self.protobuf_data:
            return data
        proto = self.opts.protobuf()
        proto.ParseFromString(data)
        data = json_format.MessageToDict(proto)
        return data

    @post_dump
    def json_to_protobuf(self, data):
        if not self.protobuf_data:
            return data
        proto = json_format.ParseDict(data, self.opts.protobuf())
        return proto.SerializeToString()
