from collections import defaultdict
from typing import Dict

from dateutil.relativedelta import relativedelta
from marshmallow import fields, post_load

from mail.ciao.ciao.api.schemas.base import BaseSchema


class SysDatetimeSchema(BaseSchema):
    seconds = fields.Integer()
    minutes = fields.Integer()
    hours = fields.Integer()
    days = fields.Integer()
    weeks = fields.Integer()
    months = fields.Integer()
    years = fields.Integer()

    seconds_relative = fields.Boolean()
    minutes_relative = fields.Boolean()
    hours_relative = fields.Boolean()
    days_relative = fields.Boolean()
    weeks_relative = fields.Boolean()
    months_relative = fields.Boolean()
    years_relative = fields.Boolean()


class SysDatetimeRangeSchema(BaseSchema):
    """
    Example: '{"start":{"minutes":0,"minutes_relative":true},"end":{"minutes":30,"minutes_relative":true}}'
    """

    start = fields.Nested(SysDatetimeSchema, required=True)
    end = fields.Nested(SysDatetimeSchema, required=True)

    @post_load
    def build_timedelta(self, data: dict) -> relativedelta:
        start_data, end_data = data['start'], data['end']
        kwargs: Dict[str, int] = defaultdict(int)
        for multiplier, part_data in (
            (1, end_data),
            (-1, start_data),
        ):
            for key, value in part_data.items():
                if not key.endswith('_relative'):
                    kwargs[key] += multiplier * value
        return relativedelta(**kwargs)  # type: ignore


_sys_datetime_range_schema = SysDatetimeRangeSchema()


def parse_sys_datetime_range(value: str) -> relativedelta:
    return _sys_datetime_range_schema.loads(value).data
