from dataclasses import asdict

from marshmallow import ValidationError, fields, post_dump, pre_dump
from marshmallow_enum import EnumField

from alice.megamind.protos.scenarios.response_pb2 import TScenarioCommitResponse, TScenarioRunResponse
from mail.ciao.ciao.api.schemas.fields import DictStrToNested
from mail.ciao.ciao.api.schemas.megamind.analytics import AnalyticsInfoSchema
from mail.ciao.ciao.api.schemas.megamind.arguments import ArgumentsSchema
from mail.ciao.ciao.api.schemas.megamind.base import BaseMegamindSchema
from mail.ciao.ciao.api.schemas.megamind.directive import DirectiveListSchema
from mail.ciao.ciao.api.schemas.megamind.frame import SemanticFrameSchema
from mail.ciao.ciao.api.schemas.megamind.slot import dump_entities
from mail.ciao.ciao.api.schemas.megamind.state import StateSchema
from mail.ciao.ciao.core.entities.button import Button
from mail.ciao.ciao.core.entities.enums import FrameName


class ScenarioErrorSchema(BaseMegamindSchema):
    message = fields.String()
    type = fields.String()


class ButtonSchema(BaseMegamindSchema):
    title = fields.String()
    action_id = fields.String()


class TextWithButtonsSchema(BaseMegamindSchema):
    text = fields.String()
    buttons = fields.Nested(ButtonSchema, many=True)

    @pre_dump
    def prepare_buttons(self, data):
        buttons = data.pop('buttons', {})
        if not isinstance(buttons, dict):
            raise ValidationError('Buttons must be dict.', 'buttons')
        data['buttons'] = [
            {
                **asdict(value),
                'action_id': key,
            }
            for key, value in buttons.items()
        ]
        return data


class CardSchema(BaseMegamindSchema):
    text = fields.String()
    text_with_buttons = fields.Nested(TextWithButtonsSchema)


class ActionButtonSchema(BaseMegamindSchema):
    action_button = fields.Nested(ButtonSchema)


class LayoutSchema(BaseMegamindSchema):
    output_speech = fields.String()
    cards = fields.Nested(CardSchema, many=True)
    should_listen = fields.Boolean()
    contains_sensitive_data = fields.Boolean()
    suggest_buttons = fields.Nested(ActionButtonSchema, many=True)

    @pre_dump
    def prepare_suggests(self, data):
        suggest_buttons = data.pop('suggest_buttons', {})
        if not isinstance(suggest_buttons, dict):
            raise ValidationError('Buttons must be dict.', 'suggest_buttons')
        data['suggest_buttons'] = [
            {
                'action_button': {
                    **asdict(value),
                    'action_id': key,
                }
            }
            for key, value in suggest_buttons.items()
        ]
        return data


class NLUHintSchema(BaseMegamindSchema):
    frame_name = EnumField(FrameName, by_value=True)


class FrameActionSchema(BaseMegamindSchema):
    nlu_hint = fields.Nested(NLUHintSchema)
    directives = fields.Nested(DirectiveListSchema)

    @pre_dump
    def prepare_directives(self, data):
        if isinstance(data, FrameName):
            return {'nlu_hint': {'frame_name': data}}
        elif isinstance(data, Button):
            return {'directives': {'list': [data]}}
        raise ValidationError('Unexpected frame action.')


class ResponseBodySchema(BaseMegamindSchema):
    analytics_info = fields.Nested(AnalyticsInfoSchema)
    semantic_frame = fields.Nested(SemanticFrameSchema)
    state = fields.Nested(StateSchema)
    layout = fields.Nested(LayoutSchema)
    expects_request = fields.Boolean()
    frame_actions = DictStrToNested(FrameActionSchema)

    @post_dump
    def dump_entities(self, data):
        slots = data.get('semantic_frame', {}).get('slots', [])
        data['entities'] = dump_entities(slots)
        return data


class CommitCandidateSchema(BaseMegamindSchema):
    response_body = fields.Nested(ResponseBodySchema)
    arguments = fields.Nested(ArgumentsSchema)


class FeaturesSchema(BaseMegamindSchema):
    is_irrelevant = fields.Boolean()


class ErrorSchema(BaseMegamindSchema):
    message = fields.String()
    type = fields.String()


class RunResponseSchema(BaseMegamindSchema):
    response_body = fields.Nested(ResponseBodySchema)
    commit_candidate = fields.Nested(CommitCandidateSchema)
    features = fields.Nested(FeaturesSchema)
    error = fields.Nested(ErrorSchema)

    class Meta:
        strict = True
        protobuf = TScenarioRunResponse


class CommitResponseSchema(BaseMegamindSchema):
    success = fields.Nested(BaseMegamindSchema)
    error = fields.Nested(ScenarioErrorSchema)

    class Meta:
        strict = True
        protobuf = TScenarioCommitResponse


run_response_json_schema = RunResponseSchema(context={'protobuf': False})
run_response_protobuf_schema = RunResponseSchema(context={'protobuf': True})

commit_response_json_schema = CommitResponseSchema(context={'protobuf': False})
commit_response_protobuf_schema = CommitResponseSchema(context={'protobuf': True})
