from enum import Enum
from typing import Any, Callable, Dict, Type, Union

from marshmallow import ValidationError, fields, post_load

from mail.ciao.ciao.api.schemas.megamind.base import BaseMegamindSchema
from mail.ciao.ciao.api.schemas.megamind.parsers.sys_date import parse_sys_date
from mail.ciao.ciao.api.schemas.megamind.parsers.sys_datetime_range import parse_sys_datetime_range
from mail.ciao.ciao.api.schemas.megamind.parsers.sys_time import parse_sys_time
from mail.ciao.ciao.api.schemas.megamind.parsers.sys_weekdays import parse_sys_weekdays
from mail.ciao.ciao.core.entities.enums import CreateEventAllDay, DateEnum, EventListSingleEvent, SysSlotType, YesNo

ALICE_LANGUAGE_RU = 1

_ENUM_SLOT_TO_NAME: Dict[Type[Enum], str] = {
    YesNo: 'yes_no',
}


# Temporary solution. Once begemot supports links to granet entities this can be removed.
# TODO: remove this after <insert ticket here> is resolved.  # TODO: insert ticket there
_ENUM_PHRASES: Dict[str, Dict[str, str]] = {
    _ENUM_SLOT_TO_NAME[YesNo]: {
        str(YesNo.YES.value): (
            'да',
            'ага',
            'давай',
            'точно',
            'конечно',
        ),
        str(YesNo.NO.value): (
            'нет',
            'не-а',
            'не надо',
        ),
    }
}


_ENUM_NAME_TO_SLOT: Dict[str, Type[Enum]] = {
    value: key
    for key, value in _ENUM_SLOT_TO_NAME.items()
}


_SLOT_PARSERS: Dict[SysSlotType, Callable] = {
    SysSlotType.DATE: parse_sys_date,
    SysSlotType.DATETIME_RANGE: parse_sys_datetime_range,
    SysSlotType.STRING: str,
    SysSlotType.TIME: parse_sys_time,
    SysSlotType.WEEKDAYS: parse_sys_weekdays,
    SysSlotType.CUSTOM_DATE_ENUM: DateEnum,
    SysSlotType.CUSTOM_EVENT_LIST_SINGLE_EVENT: EventListSingleEvent,
    SysSlotType.CUSTOM_CREATE_EVENT_ALL_DAY: CreateEventAllDay,
}


def parse_slot(slot_type: Union[Type[Enum], SysSlotType], slot_value: str) -> Any:
    if isinstance(slot_type, SysSlotType):
        if slot_type not in _SLOT_PARSERS:
            raise ValidationError(f'Unknown slot type "{slot_type}".')
        try:
            return _SLOT_PARSERS[slot_type](slot_value)
        except ValueError:
            raise ValidationError(f'Invalid slot value "{slot_value}" for type "{slot_type}".')
    else:
        try:
            return slot_type(slot_value)
        except ValueError:
            raise ValidationError(f'Invalid enum value "{slot_value}" for "{slot_type}".')


class SlotTypeField(fields.Field):
    def _deserialize(self, value, attr, data):
        if value in _ENUM_NAME_TO_SLOT:
            return _ENUM_NAME_TO_SLOT[value]
        try:
            return SysSlotType(value)
        except ValueError:
            raise ValidationError(f'Unknown slot type "{value}".')

    def _serialize(self, value, attr, obj):
        if isinstance(value, SysSlotType):
            return value.value
        else:
            return _ENUM_SLOT_TO_NAME[value]


class SlotSchema(BaseMegamindSchema):
    name = fields.String(required=True)
    type = SlotTypeField(required=True)
    value = fields.String(required=True)
    accepted_types = fields.List(SlotTypeField)
    is_requested = fields.Boolean()

    @post_load
    def add_parsed(self, data):
        data['parsed'] = parse_slot(data['type'], data['value'])
        return data


def dump_entities(slots):
    entity_names = set()
    for slot in slots:
        for type in slot.get('accepted_types', []):
            if type in _ENUM_NAME_TO_SLOT:
                entity_names.add(type)
    entities = []
    for name in entity_names:
        entities.append({
            'name': name,
            'items': {
                value: {
                    'instances': [
                        {'language': ALICE_LANGUAGE_RU, 'phrase': phrase}
                        for phrase in phrases
                    ]
                }
                for value, phrases in _ENUM_PHRASES[name].items()
            }
        })
    return entities
